<?php
/*
 * @package   bfNetwork
 * @copyright Copyright (C) 2011,2012,2013,2014,2015,2016,2017,2018,2019,2020,2021,2022,2023 Blue Flame Digital Solutions Ltd. All rights reserved.
 * @license   GNU General Public License version 3 or later
 *
 * @see       https://mySites.guru/
 * @see       https://www.phil-taylor.com/
 *
 * @author    Phil Taylor / Blue Flame Digital Solutions Limited.
 *
 * bfNetwork is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * bfNetwork is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this package.  If not, see http://www.gnu.org/licenses/
 *
 * If you have any questions regarding this code, please contact phil@phil-taylor.com
 */

use Joomla\CMS\Filesystem\File;
use Joomla\CMS\Filesystem\Folder;
use Joomla\CMS\Filesystem\Path;

class Bf_Filesystem
{
    /**
     * @param string $file
     * @param string $buffer
     *
     * @return bool
     */
    public static function _write($file, $buffer)
    {
        if (! file_exists(dirname($file))) {
            Folder::create(dirname($file));
        }

        Path::setPermissions($file, '0644');

        return File::write($file, $buffer);
    }

    /**
     * @param string $filter
     * @param bool   $recurse
     * @param bool   $fullpath
     *
     * @return array
     */
    public static function readDirectory($path, $filter = '.', $recurse = false, $fullpath = false)
    {
        $arr = [];
        if (! @is_dir($path)) {
            return $arr;
        }
        $handle = opendir($path);

        while ($file = readdir($handle)) {
            $dir   = self::pathName($path . '/' . $file, false);
            $isDir = is_dir($dir);
            if (('.' != $file) && ('..' != $file)) {
                if (preg_match("/$filter/", $file)) {
                    if ($fullpath) {
                        $arr[] = trim((string) self::pathName($path . '/' . $file, false));
                    } else {
                        $arr[] = trim($file);
                    }
                }
                if ($recurse && $isDir) {
                    $arr2 = self::readDirectory($dir, $filter, $recurse, $fullpath);
                    $arr  = array_merge($arr, $arr2);
                }
            }
        }
        closedir($handle);
        asort($arr);

        return $arr;
    }

    /**
     * @param bool $p_addtrailingslash
     *
     * @return mixed|string
     */
    public static function pathName($p_path, $p_addtrailingslash = true)
    {
        $retval = '';

        $isWin = (str_starts_with(\PHP_OS, 'WIN'));

        if ($isWin) {
            $retval = str_replace('/', '\\', (string) $p_path);
            if ($p_addtrailingslash) {
                if (! str_ends_with($retval, '\\')) {
                    $retval .= '\\';
                }
            }

            // Check if UNC path
            $unc = str_starts_with($retval, '\\\\') ? 1 : 0;

            // Remove double \\
            $retval = str_replace('\\\\', '\\', $retval);

            // If UNC path, we have to add one \ in front or everything breaks!
            if (1 == $unc) {
                $retval = '\\' . $retval;
            }
        } else {
            $retval = str_replace('\\', '/', (string) $p_path);
            if ($p_addtrailingslash) {
                if (! str_ends_with($retval, '/')) {
                    $retval .= '/';
                }
            }

            // Check if UNC path
            $unc = str_starts_with($retval, '//') ? 1 : 0;

            // Remove double //
            $retval = str_replace('//', '/', $retval);

            // If UNC path, we have to add one / in front or everything breaks!
            if (1 == $unc) {
                $retval = '/' . $retval;
            }
        }

        return $retval;
    }

    /**
     * returns the directories in the path if append path is set then this path will appended to the results.
     *
     * @param string      $path
     * @param bool|string $appendPath
     *
     * @return array
     */
    public static function getDirectories($path, $appendPath = false)
    {
        if (is_dir($path)) {
            $contents = scandir($path); //open directory and get contents
            if (is_array($contents)) { //it found files
                $returnDirs = false;
                foreach ($contents as $dir) {
                    //validate that this is a directory
                    if (is_dir($path . '/' . $dir) &&
                        '.' != $dir                && '..' != $dir && '.svn' != $dir
                    ) {
                        $returnDirs[] = $appendPath . $dir;
                    }
                }

                if ($returnDirs) {
                    return $returnDirs;
                }
            }
        }
    }

    /**
     * adds a complete directory path eg: /my/own/path will create >my >>own >>>path.
     *
     * @param string $base
     * @param string $path
     *
     * @return bool
     */
    public static function makeRecursive($base, $path)
    {
        $pathArray = explode('/', $path);
        if (is_array($pathArray)) {
            $strPath = null;
            foreach ($pathArray as $path) {
                if (! empty($path)) {
                    $strPath .= '/' . $path;
                    if (! is_dir($base . $strPath)) {
                        if (! self::make($base . $strPath)) {
                            return false;
                        }
                    }
                }
            }

            return true;
        }
    }

    /**
     * this is getting a little extreme i know but it will help out later when we want to keep updated indexes for right
     * now, not much.
     *
     * @param string $path
     *
     * @return bool
     */
    public static function make($path)
    {
        return mkdir($path, 0777);
    }

    /**
     * deletes a directory recursively Bf_Filesystem::deleteRecursive(JPATH_ROOT .'/tmp', TRUE);.
     *
     * @param string $target
     * @param bool   $ignoreWarnings
     * @param array  $msg
     *
     * @return bool|array
     */
    public static function deleteRecursive($target, $ignoreWarnings = false, $msg = [])
    {
        $exceptions = ['.', '..'];
        if (! $sourceDir = @opendir($target)) {
            if (true !== $ignoreWarnings) {
                $msg['result']   = 'failure';
                $msg['errors'][] = $target;

                return $msg;
            }
        }

        if (! $sourceDir) {
            return false;
        }

        while (false !== ($sibling = readdir($sourceDir))) {
            if (! in_array($sibling, $exceptions)) {
                $object = str_replace('//', '/', $target . '/' . $sibling);
                if (is_dir($object)) {
                    $msg = self::deleteRecursive($object, $ignoreWarnings, $msg);
                }

                if (is_file($object)) {
                    bfLog::log('Deleting ' . $object);

                    $result = unlink($object);
                    if ($result) {
                        $msg['deleted_files'][] = $object;
                    }
                    if (! $result) {
                        $msg['errors'][] = $object;
                    }
                }
            }
        }

        closedir($sourceDir);

        if (is_dir($target)) {
            bfLog::log('Deleting ' . $target);

            if ($result = rmdir($target)) {
                $msg['deleted_folders'][] = $target;
                $msg['result']            = 'success';
            } else {
                bfLog::log(sprintf('Deleting %S FAILED', $target));
                $msg['result'] = 'failure';
            }
        }

        return $msg;
    }
}
