<?php
/*
 * @package   bfNetwork
 * @copyright Copyright (C) 2011,2012,2013,2014,2015,2016,2017,2018,2019,2020,2021,2022,2023 Blue Flame Digital Solutions Ltd. All rights reserved.
 * @license   GNU General Public License version 3 or later
 *
 * @see       https://mySites.guru/
 * @see       https://www.phil-taylor.com/
 *
 * @author    Phil Taylor / Blue Flame Digital Solutions Limited.
 *
 * bfNetwork is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * bfNetwork is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this package.  If not, see http://www.gnu.org/licenses/
 *
 * If you have any questions regarding this code, please contact phil@phil-taylor.com
 */

/*
 * Include Crypt_Random.
 */
// the class_exists() will only be called if the crypt_random_string function hasn't been defined and
// will trigger a call to __autoload() if you're wanting to auto-load classes
// call function_exists() a second time to stop the include_once from being called outside
// of the auto loader
if (! function_exists('crypt_random_string')) {
    include_once 'Random.php';
}

/*
 * Include Crypt_Hash
 */
if (! class_exists('Crypt_Hash')) {
    include_once 'Hash.php';
}

/**#@+
 * @access public
 * @see Crypt_RSA::encrypt()
 * @see Crypt_RSA::decrypt()
 */
/*
 * Use {@link http://en.wikipedia.org/wiki/Optimal_Asymmetric_Encryption_Padding Optimal Asymmetric Encryption Padding}
 * (OAEP) for encryption / decryption.
 *
 * Uses sha1 by default.
 *
 * @see Crypt_RSA::setHash()
 * @see Crypt_RSA::setMGFHash()
 */
define('CRYPT_RSA_ENCRYPTION_OAEP', 1);
/*
 * Use PKCS#1 padding.
 *
 * Although CRYPT_RSA_ENCRYPTION_OAEP offers more security, including PKCS#1 padding is necessary for purposes of backwards
 * compatibility with protocols (like SSH-1) written before OAEP's introduction.
 */
define('CRYPT_RSA_ENCRYPTION_PKCS1', 2);
/**#@-*/

/**#@+
 * @access public
 * @see Crypt_RSA::sign()
 * @see Crypt_RSA::verify()
 * @see Crypt_RSA::setHash()
 */
/*
 * Use the Probabilistic Signature Scheme for signing
 *
 * Uses sha1 by default.
 *
 * @see Crypt_RSA::setSaltLength()
 * @see Crypt_RSA::setMGFHash()
 */
define('CRYPT_RSA_SIGNATURE_PSS', 1);
/*
 * Use the PKCS#1 scheme by default.
 *
 * Although CRYPT_RSA_SIGNATURE_PSS offers more security, including PKCS#1 signing is necessary for purposes of backwards
 * compatibility with protocols (like SSH-2) written before PSS's introduction.
 */
define('CRYPT_RSA_SIGNATURE_PKCS1', 2);
/**#@-*/

/**#@+
 * @access private
 * @see Crypt_RSA::createKey()
 */
/*
 * ASN1 Integer
 */
define('CRYPT_RSA_ASN1_INTEGER', 2);
/*
 * ASN1 Bit String
 */
define('CRYPT_RSA_ASN1_BITSTRING', 3);
/*
 * ASN1 Octet String
 */
define('CRYPT_RSA_ASN1_OCTETSTRING', 4);
/*
 * ASN1 Object Identifier
 */
define('CRYPT_RSA_ASN1_OBJECT', 6);
/*
 * ASN1 Sequence (with the constucted bit set)
 */
define('CRYPT_RSA_ASN1_SEQUENCE', 48);
/**#@-*/

/**#@+
 * @access private
 * @see Crypt_RSA::Crypt_RSA()
 */
/*
 * To use the pure-PHP implementation
 */
define('CRYPT_RSA_MODE_INTERNAL', 1);
/*
 * To use the OpenSSL library
 *
 * (if enabled; otherwise, the internal implementation will be used)
 */
define('CRYPT_RSA_MODE_OPENSSL', 2);
/**#@-*/

/*
 * Default openSSL configuration file.
 */
define('CRYPT_RSA_OPENSSL_CONFIG', __DIR__ . '/../openssl.cnf');

/**#@+
 * @access public
 * @see Crypt_RSA::createKey()
 * @see Crypt_RSA::setPrivateKeyFormat()
 */
/*
 * PKCS#1 formatted private key
 *
 * Used by OpenSSH
 */
define('CRYPT_RSA_PRIVATE_FORMAT_PKCS1', 0);
/*
 * PuTTY formatted private key
 */
define('CRYPT_RSA_PRIVATE_FORMAT_PUTTY', 1);
/*
 * XML formatted private key
 */
define('CRYPT_RSA_PRIVATE_FORMAT_XML', 2);
/*
 * PKCS#8 formatted private key
 */
define('CRYPT_RSA_PRIVATE_FORMAT_PKCS8', 3);
/**#@-*/

/**#@+
 * @access public
 * @see Crypt_RSA::createKey()
 * @see Crypt_RSA::setPublicKeyFormat()
 */
/*
 * Raw public key
 *
 * An array containing two Math_BigInteger objects.
 *
 * The exponent can be indexed with any of the following:
 *
 * 0, e, exponent, publicExponent
 *
 * The modulus can be indexed with any of the following:
 *
 * 1, n, modulo, modulus
 */
define('CRYPT_RSA_PUBLIC_FORMAT_RAW', 3);
/*
 * PKCS#1 formatted public key (raw)
 *
 * Used by File/X509.php
 *
 * Has the following header:
 *
 * -----BEGIN RSA PUBLIC KEY-----
 *
 * Analogous to ssh-keygen's pem format (as specified by -m)
 */
define('CRYPT_RSA_PUBLIC_FORMAT_PKCS1', 4);
define('CRYPT_RSA_PUBLIC_FORMAT_PKCS1_RAW', 4);
/*
 * XML formatted public key
 */
define('CRYPT_RSA_PUBLIC_FORMAT_XML', 5);
/*
 * OpenSSH formatted public key
 *
 * Place in $HOME/.ssh/authorized_keys
 */
define('CRYPT_RSA_PUBLIC_FORMAT_OPENSSH', 6);
/*
 * PKCS#1 formatted public key (encapsulated)
 *
 * Used by PHP's openssl_public_encrypt() and openssl's rsautl (when -pubin is set)
 *
 * Has the following header:
 *
 * -----BEGIN PUBLIC KEY-----
 *
 * Analogous to ssh-keygen's pkcs8 format (as specified by -m). Although PKCS8
 * is specific to private keys it's basically creating a DER-encoded wrapper
 * for keys. This just extends that same concept to public keys (much like ssh-keygen)
 */
define('CRYPT_RSA_PUBLIC_FORMAT_PKCS8', 7);
/**#@-*/

/**
 * Pure-PHP PKCS#1 compliant implementation of RSA.
 *
 * @author  Jim Wigginton <terrafrost@php.net>
 */
class Crypt_RSA implements \Stringable
{
    /**
     * Precomputed Zero.
     *
     * @var array
     */
    public $zero;

    /**
     * Precomputed One.
     *
     * @var array
     */
    public $one;

    /**
     * Private Key Format.
     *
     * @var int
     */
    public $privateKeyFormat = CRYPT_RSA_PRIVATE_FORMAT_PKCS1;

    /**
     * Public Key Format.
     *
     * @var int
     */
    public $publicKeyFormat = CRYPT_RSA_PUBLIC_FORMAT_PKCS8;

    /**
     * Modulus (ie. n).
     *
     * @var Math_BigInteger
     */
    public $modulus;

    /**
     * Modulus length.
     *
     * @var Math_BigInteger
     */
    public $k;

    /**
     * Exponent (ie. e or d).
     *
     * @var Math_BigInteger
     */
    public $exponent;

    /**
     * Primes for Chinese Remainder Theorem (ie. p and q).
     *
     * @var array
     */
    public $primes;

    /**
     * Exponents for Chinese Remainder Theorem (ie. dP and dQ).
     *
     * @var array
     */
    public $exponents;

    /**
     * Coefficients for Chinese Remainder Theorem (ie. qInv).
     *
     * @var array
     */
    public $coefficients;

    /**
     * Hash name.
     *
     * @var string
     */
    public $hashName;

    /**
     * Hash function.
     *
     * @var Crypt_Hash
     */
    public $hash;

    /**
     * Length of hash function output.
     *
     * @var int
     */
    public $hLen;

    /**
     * Length of salt.
     *
     * @var int
     */
    public $sLen;

    /**
     * Hash function for the Mask Generation Function.
     *
     * @var Crypt_Hash
     */
    public $mgfHash;

    /**
     * Length of MGF hash function output.
     *
     * @var int
     */
    public $mgfHLen;

    /**
     * Encryption mode.
     *
     * @var int
     */
    public $encryptionMode = CRYPT_RSA_ENCRYPTION_OAEP;

    /**
     * Signature mode.
     *
     * @var int
     */
    public $signatureMode = CRYPT_RSA_SIGNATURE_PSS;

    /**
     * Public Exponent.
     */
    public $publicExponent = false;

    /**
     * Password.
     *
     * @var string
     */
    public $password = false;

    /**
     * Components.
     *
     * For use with parsing XML formatted keys.  PHP's XML Parser functions use utilized - instead of PHP's DOM
     * functions - because PHP's XML Parser functions work on PHP4 whereas PHP's DOM functions - although surperior -
     * don't.
     *
     * @see Crypt_RSA::_start_element_handler()
     *
     * @var array
     */
    public $components = [];

    /**
     * Current String.
     *
     * For use with parsing XML formatted keys.
     *
     * @see Crypt_RSA::_character_handler()
     * @see Crypt_RSA::_stop_element_handler()
     */
    public $current;

    /**
     * OpenSSL configuration file name.
     *
     * Set to null to use system configuration file.
     *
     * @see Crypt_RSA::createKey()
     *
     * @Access public
     */
    public $configFile;

    /**
     * Public key comment field.
     *
     * @var string
     */
    public $comment = 'phpseclib-generated-key';

    /**
     * The constructor.
     *
     * If you want to make use of the openssl extension, you'll need to set the mode manually, yourself.  The reason
     * Crypt_RSA doesn't do it is because OpenSSL doesn't fail gracefully.  openssl_pkey_new(), in particular, requires
     * openssl.cnf be present somewhere and, unfortunately, the only real way to find out is too late.
     *
     * @return Crypt_RSA
     */
    public function __construct()
    {
        if (! class_exists('Math_BigInteger')) {
            include_once 'Math/BigInteger.php';
        }

        $this->configFile = CRYPT_RSA_OPENSSL_CONFIG;

        if (preg_match('/main-hosting.eu/ism', (function_exists('gethostname') ? gethostname() : php_uname('n')))) {
            if (! defined('CRYPT_RSA_MODE')) {
                define('CRYPT_RSA_MODE', CRYPT_RSA_MODE_INTERNAL);
                define('MATH_BIGINTEGER_OPENSSL_DISABLE', true);
            }
        }

        if (! defined('CRYPT_RSA_MODE')) {
            switch (true) {
                // Math/BigInteger's openssl requirements are a little less stringent than Crypt/RSA's. in particular,
                // Math/BigInteger doesn't require an openssl.cfg file whereas Crypt/RSA does. so if Math/BigInteger
                // can't use OpenSSL it can be pretty trivially assumed, then, that Crypt/RSA can't either.
                case defined('MATH_BIGINTEGER_OPENSSL_DISABLE'):
                    define('CRYPT_RSA_MODE', CRYPT_RSA_MODE_INTERNAL);
                    break;
                    // openssl_pkey_get_details - which is used in the only place Crypt/RSA.php uses OpenSSL - was introduced in PHP 5.2.0
                case ! function_exists('openssl_pkey_get_details'):
                    define('CRYPT_RSA_MODE', CRYPT_RSA_MODE_INTERNAL);
                    break;
                case ! function_exists('phpinfo'):
                    define('CRYPT_RSA_MODE', CRYPT_RSA_MODE_INTERNAL);
                    define('MATH_BIGINTEGER_OPENSSL_DISABLE', true);
                    break;
                case extension_loaded('openssl') && version_compare(\PHP_VERSION, '4.2.0', '>=') && file_exists($this->configFile):
                    // some versions of XAMPP have mismatched versions of OpenSSL which causes it not to work
                    ob_start();
                    @phpinfo();
                    $content = ob_get_contents();
                    ob_end_clean();

                    preg_match_all('#OpenSSL (Header|Library) Version(.*)#im', $content, $matches);

                    $versions = [];
                    if (! empty($matches[1])) {
                        for ($i = 0; $i < count($matches[1]); ++$i) {
                            $fullVersion = trim(str_replace('=>', '', strip_tags((string) $matches[2][$i])));

                            // Remove letter part in OpenSSL version
                            if (! preg_match('/(\d+\.\d+\.\d+)/i', $fullVersion, $m)) {
                                $versions[$matches[1][$i]] = $fullVersion;
                            } else {
                                $versions[$matches[1][$i]] = $m[0];
                            }
                        }
                    }

                    // it doesn't appear that OpenSSL versions were reported upon until PHP 5.3+
                    switch (true) {
                        case ! isset($versions['Header']):
                        case ! isset($versions['Library']):
                        case $versions['Header'] == $versions['Library']:
                            define('CRYPT_RSA_MODE', CRYPT_RSA_MODE_OPENSSL);
                            break;
                        default:
                            define('CRYPT_RSA_MODE', CRYPT_RSA_MODE_INTERNAL);
                            define('MATH_BIGINTEGER_OPENSSL_DISABLE', true);
                    }
                    break;
                default:
                    define('CRYPT_RSA_MODE', CRYPT_RSA_MODE_INTERNAL);
            }
        }

        $this->zero = new Math_BigInteger();
        $this->one  = new Math_BigInteger(1);

        $this->hash     = new Crypt_Hash('sha1');
        $this->hLen     = $this->hash->getLength();
        $this->hashName = 'sha1';
        $this->mgfHash  = new Crypt_Hash('sha1');
        $this->mgfHLen  = $this->mgfHash->getLength();
    }

    /**
     * Create public / private key pair.
     *
     * Returns an array with the following three elements:
     *  - 'privatekey': The private key.
     *  - 'publickey':  The public key.
     *  - 'partialkey': A partially computed key (if the execution time exceeded $timeout).
     *                  Will need to be passed back to Crypt_RSA::createKey() as the third parameter for further processing.
     *
     * @param optional Integer         $bits
     * @param optional Integer         $timeout
     * @param optional Math_BigInteger $partial
     */
    public function createKey($bits = 1024, $timeout = false, $partial = [])
    {
        $primes       = [];
        $min          = null;
        $max          = null;
        $coefficients = [];
        $lcm          = [];
        $exponents    = [];
        if (! defined('CRYPT_RSA_EXPONENT')) {
            // http://en.wikipedia.org/wiki/65537_%28number%29
            define('CRYPT_RSA_EXPONENT', '65537');
        }
        // per <http://cseweb.ucsd.edu/~hovav/dist/survey.pdf#page=5>, this number ought not result in primes smaller
        // than 256 bits. as a consequence if the key you're trying to create is 1024 bits and you've set CRYPT_RSA_SMALLEST_PRIME
        // to 384 bits then you're going to get a 384 bit prime and a 640 bit prime (384 + 1024 % 384). at least if
        // CRYPT_RSA_MODE is set to CRYPT_RSA_MODE_INTERNAL. if CRYPT_RSA_MODE is set to CRYPT_RSA_MODE_OPENSSL then
        // CRYPT_RSA_SMALLEST_PRIME is ignored (ie. multi-prime RSA support is more intended as a way to speed up RSA key
        // generation when there's a chance neither gmp nor OpenSSL are installed)
        if (! defined('CRYPT_RSA_SMALLEST_PRIME')) {
            define('CRYPT_RSA_SMALLEST_PRIME', 4096);
        }

        // OpenSSL uses 65537 as the exponent and requires RSA keys be 384 bits minimum
        if (CRYPT_RSA_MODE == CRYPT_RSA_MODE_OPENSSL && $bits >= 384 && CRYPT_RSA_EXPONENT == 65537) {
            $config = [];
            if (isset($this->configFile)) {
                $config['config'] = $this->configFile;
            }
            $rsa = openssl_pkey_new([
                'private_key_bits' => $bits,
            ] + $config);
            openssl_pkey_export($rsa, $privatekey, null, $config);
            $publickey = openssl_pkey_get_details($rsa);
            $publickey = $publickey['key'];

            $privatekey = call_user_func_array(
                $this->_convertPrivateKey(...),
                array_values($this->_parseKey($privatekey, CRYPT_RSA_PRIVATE_FORMAT_PKCS1))
            );
            $publickey  = call_user_func_array(
                $this->_convertPublicKey(...),
                array_values($this->_parseKey($publickey, CRYPT_RSA_PUBLIC_FORMAT_PKCS1))
            );

            // clear the buffer of error strings stemming from a minimalistic openssl.cnf
            while (false !== openssl_error_string());

            return [
                'privatekey' => $privatekey,
                'publickey'  => $publickey,
                'partialkey' => false,
            ];
        }

        static $e;
        if (! isset($e)) {
            $e = new Math_BigInteger(CRYPT_RSA_EXPONENT);
        }
        $generateMinMax = $this->_generateMinMax($bits);

        extract($generateMinMax);
        $absoluteMin = $min;
        $temp        = $bits >> 1; // divide by two to see how many bits P and Q would be
        if ($temp > CRYPT_RSA_SMALLEST_PRIME) {
            $num_primes = floor($bits / CRYPT_RSA_SMALLEST_PRIME);
            $temp       = CRYPT_RSA_SMALLEST_PRIME;
        } else {
            $num_primes = 2;
        }
        $generateMinMax2 = $this->_generateMinMax($temp + $bits % $temp);
        extract($generateMinMax2);
        $finalMax        = $max;
        $generateMinMax3 = $this->_generateMinMax($temp);
        extract($generateMinMax3);

        $generator = new Math_BigInteger();

        $n = $this->one->copy();
        if (! empty($partial)) {
            $unserialize = unserialize($partial);
            extract($unserialize);
        } else {
            $exponents = $coefficients = $primes = [];
            $lcm       = [
                'top'    => $this->one->copy(),
                'bottom' => false,
            ];
        }

        $start = time();
        $i0    = count($primes) + 1;

        do {
            for ($i = $i0; $i <= $num_primes; ++$i) {
                if (false !== $timeout) {
                    $timeout -= time() - $start;
                    $start = time();
                    if ($timeout <= 0) {
                        return [
                            'privatekey' => '',
                            'publickey'  => '',
                            'partialkey' => serialize([
                                'primes'       => $primes,
                                'coefficients' => $coefficients,
                                'lcm'          => $lcm,
                                'exponents'    => $exponents,
                            ]),
                        ];
                    }
                }

                if ($i == $num_primes) {
                    [$min, $temp] = $absoluteMin->divide($n);
                    if (! $temp->equals($this->zero)) {
                        $min = $min->add($this->one); // ie. ceil()
                    }
                    $primes[$i] = $generator->randomPrime($min, $finalMax, $timeout);
                } else {
                    $primes[$i] = $generator->randomPrime($min, $max, $timeout);
                }

                if (false === $primes[$i]) { // if we've reached the timeout
                    if (count($primes) > 1) {
                        $partialkey = '';
                    } else {
                        array_pop($primes);
                        $partialkey = serialize([
                            'primes'       => $primes,
                            'coefficients' => $coefficients,
                            'lcm'          => $lcm,
                            'exponents'    => $exponents,
                        ]);
                    }

                    return [
                        'privatekey' => '',
                        'publickey'  => '',
                        'partialkey' => $partialkey,
                    ];
                }

                // the first coefficient is calculated differently from the rest
                // ie. instead of being $primes[1]->modInverse($primes[2]), it's $primes[2]->modInverse($primes[1])
                if ($i > 2) {
                    $coefficients[$i] = $n->modInverse($primes[$i]);
                }

                $n = $n->multiply($primes[$i]);

                $temp = $primes[$i]->subtract($this->one);

                // textbook RSA implementations use Euler's totient function instead of the least common multiple.
                // see http://en.wikipedia.org/wiki/Euler%27s_totient_function
                $lcm['top']    = $lcm['top']->multiply($temp);
                $lcm['bottom'] = false === $lcm['bottom'] ? $temp : $lcm['bottom']->gcd($temp);

                $exponents[$i] = $e->modInverse($temp);
            }

            [$temp]     = $lcm['top']->divide($lcm['bottom']);
            $gcd        = $temp->gcd($e);
            $i0         = 1;
        } while (! $gcd->equals($this->one));

        $d = $e->modInverse($temp);

        $coefficients[2] = $primes[2]->modInverse($primes[1]);

        // from <http://tools.ietf.org/html/rfc3447#appendix-A.1.2>:
        // RSAPrivateKey ::= SEQUENCE {
        //     version           Version,
        //     modulus           INTEGER,  -- n
        //     publicExponent    INTEGER,  -- e
        //     privateExponent   INTEGER,  -- d
        //     prime1            INTEGER,  -- p
        //     prime2            INTEGER,  -- q
        //     exponent1         INTEGER,  -- d mod (p-1)
        //     exponent2         INTEGER,  -- d mod (q-1)
        //     coefficient       INTEGER,  -- (inverse of q) mod p
        //     otherPrimeInfos   OtherPrimeInfos OPTIONAL
        // }

        return [
            'privatekey' => $this->_convertPrivateKey($n, $e, $d, $primes, $exponents, $coefficients),
            'publickey'  => $this->_convertPublicKey($n, $e),
            'partialkey' => false,
        ];
    }

    /**
     * Break a public or private key down into its constituant components.
     *
     * @see _convertPublicKey()
     * @see _convertPrivateKey()
     *
     * @param string $key
     * @param int    $type
     *
     * @return array
     */
    public function _parseKey($key, $type)
    {
        $components = [];
        $length     = null;
        if (CRYPT_RSA_PUBLIC_FORMAT_RAW != $type && ! is_string($key)) {
            return false;
        }

        switch ($type) {
            case CRYPT_RSA_PUBLIC_FORMAT_RAW:
                if (! is_array($key)) {
                    return false;
                }
                $components = [];
                switch (true) {
                    case isset($key['e']):
                        $components['publicExponent'] = $key['e']->copy();
                        break;
                    case isset($key['exponent']):
                        $components['publicExponent'] = $key['exponent']->copy();
                        break;
                    case isset($key['publicExponent']):
                        $components['publicExponent'] = $key['publicExponent']->copy();
                        break;
                    case isset($key[0]):
                        $components['publicExponent'] = $key[0]->copy();
                }
                switch (true) {
                    case isset($key['n']):
                        $components['modulus'] = $key['n']->copy();
                        break;
                    case isset($key['modulo']):
                        $components['modulus'] = $key['modulo']->copy();
                        break;
                    case isset($key['modulus']):
                        $components['modulus'] = $key['modulus']->copy();
                        break;
                    case isset($key[1]):
                        $components['modulus'] = $key[1]->copy();
                }

                return isset($components['modulus']) && isset($components['publicExponent']) ? $components : false;
            case CRYPT_RSA_PRIVATE_FORMAT_PKCS1:
            case CRYPT_RSA_PRIVATE_FORMAT_PKCS8:
            case CRYPT_RSA_PUBLIC_FORMAT_PKCS1:
                /* Although PKCS#1 proposes a format that public and private keys can use, encrypting them is
                   "outside the scope" of PKCS#1.  PKCS#1 then refers you to PKCS#12 and PKCS#15 if you're wanting to
                   protect private keys, however, that's not what OpenSSL* does.  OpenSSL protects private keys by adding
                   two new "fields" to the key - DEK-Info and Proc-Type.  These fields are discussed here:

                   http://tools.ietf.org/html/rfc1421#section-4.6.1.1
                   http://tools.ietf.org/html/rfc1421#section-4.6.1.3

                   DES-EDE3-CBC as an algorithm, however, is not discussed anywhere, near as I can tell.
                   DES-CBC and DES-EDE are discussed in RFC1423, however, DES-EDE3-CBC isn't, nor is its key derivation
                   function.  As is, the definitive authority on this encoding scheme isn't the IETF but rather OpenSSL's
                   own implementation.  ie. the implementation *is* the standard and any bugs that may exist in that
                   implementation are part of the standard, as well.

                   * OpenSSL is the de facto standard.  It's utilized by OpenSSH and other projects */
                if (preg_match('#DEK-Info: (.+),(.+)#', $key, $matches)) {
                    $iv     = pack('H*', trim($matches[2]));
                    $symkey = pack('H*', md5($this->password . substr($iv, 0, 8))); // symkey is short for symmetric key
                    $symkey .= pack('H*', md5($symkey . $this->password . substr($iv, 0, 8)));
                    // remove the Proc-Type / DEK-Info sections as they're no longer needed
                    $key        = preg_replace('#^(?:Proc-Type|DEK-Info): .*#m', '', $key);
                    $ciphertext = $this->_extractBER($key);
                    if (false === $ciphertext) {
                        $ciphertext = $key;
                    }
                    switch ($matches[1]) {
                        case 'AES-256-CBC':
                            if (! class_exists('Crypt_AES')) {
                                include_once 'Crypt/AES.php';
                            }
                            $crypto = new Crypt_AES();
                            break;
                        case 'AES-128-CBC':
                            if (! class_exists('Crypt_AES')) {
                                include_once 'Crypt/AES.php';
                            }
                            $symkey = substr($symkey, 0, 16);
                            $crypto = new Crypt_AES();
                            break;
                        case 'DES-EDE3-CFB':
                            if (! class_exists('Crypt_TripleDES')) {
                                include_once 'Crypt/TripleDES.php';
                            }
                            $crypto = new Crypt_TripleDES(CRYPT_DES_MODE_CFB);
                            break;
                        case 'DES-EDE3-CBC':
                            if (! class_exists('Crypt_TripleDES')) {
                                include_once 'Crypt/TripleDES.php';
                            }
                            $symkey = substr($symkey, 0, 24);
                            $crypto = new Crypt_TripleDES();
                            break;
                        case 'DES-CBC':
                            if (! class_exists('Crypt_DES')) {
                                include_once 'Crypt/DES.php';
                            }
                            $crypto = new Crypt_DES();
                            break;
                        default:
                            return false;
                    }
                    $crypto->setKey($symkey);
                    $crypto->setIV($iv);
                    $decoded = $crypto->decrypt($ciphertext);
                } else {
                    $decoded = $this->_extractBER($key);
                }

                if (false !== $decoded) {
                    $key = $decoded;
                }

                $components = [];

                if (CRYPT_RSA_ASN1_SEQUENCE != ord($this->_string_shift($key))) {
                    return false;
                }
                if ($this->_decodeLength($key) != strlen((string) $key)) {
                    return false;
                }

                $tag = ord($this->_string_shift($key));
                /* intended for keys for which OpenSSL's asn1parse returns the following:

                    0:d=0  hl=4 l= 631 cons: SEQUENCE
                    4:d=1  hl=2 l=   1 prim:  INTEGER           :00
                    7:d=1  hl=2 l=  13 cons:  SEQUENCE
                    9:d=2  hl=2 l=   9 prim:   OBJECT            :rsaEncryption
                   20:d=2  hl=2 l=   0 prim:   NULL
                   22:d=1  hl=4 l= 609 prim:  OCTET STRING

                   ie. PKCS8 keys*/

                if (CRYPT_RSA_ASN1_INTEGER == $tag && str_starts_with((string) $key, "\x01\x00\x30")) {
                    $this->_string_shift($key, 3);
                    $tag = CRYPT_RSA_ASN1_SEQUENCE;
                }

                if (CRYPT_RSA_ASN1_SEQUENCE == $tag) {
                    $temp = $this->_string_shift($key, $this->_decodeLength($key));
                    if (CRYPT_RSA_ASN1_OBJECT != ord($this->_string_shift($temp))) {
                        return false;
                    }
                    $length = $this->_decodeLength($temp);
                    switch ($this->_string_shift($temp, $length)) {
                        case "\x2a\x86\x48\x86\xf7\x0d\x01\x01\x01": // rsaEncryption
                            break;
                        case "\x2a\x86\x48\x86\xf7\x0d\x01\x05\x03": // pbeWithMD5AndDES-CBC
                            /*
                               PBEParameter ::= SEQUENCE {
                                   salt OCTET STRING (SIZE(8)),
                                   iterationCount INTEGER }
                            */
                            if (CRYPT_RSA_ASN1_SEQUENCE != ord($this->_string_shift($temp))) {
                                return false;
                            }
                            if ($this->_decodeLength($temp) != strlen((string) $temp)) {
                                return false;
                            }
                            $this->_string_shift($temp); // assume it's an octet string
                            $salt = $this->_string_shift($temp, $this->_decodeLength($temp));
                            if (CRYPT_RSA_ASN1_INTEGER != ord($this->_string_shift($temp))) {
                                return false;
                            }
                            $this->_decodeLength($temp);
                            [, $iterationCount] = unpack('N', str_pad((string) $temp, 4, chr(0), \STR_PAD_LEFT));
                            $this->_string_shift($key); // assume it's an octet string
                            $length = $this->_decodeLength($key);
                            if (strlen((string) $key) != $length) {
                                return false;
                            }

                            if (! class_exists('Crypt_DES')) {
                                include_once 'Crypt/DES.php';
                            }
                            $crypto = new Crypt_DES();
                            $crypto->setPassword($this->password, 'pbkdf1', 'md5', $salt, $iterationCount);
                            $key = $crypto->decrypt($key);
                            if (false === $key) {
                                return false;
                            }

                            return $this->_parseKey($key, CRYPT_RSA_PRIVATE_FORMAT_PKCS1);
                        default:
                            return false;
                    }
                    /* intended for keys for which OpenSSL's asn1parse returns the following:

                        0:d=0  hl=4 l= 290 cons: SEQUENCE
                        4:d=1  hl=2 l=  13 cons:  SEQUENCE
                        6:d=2  hl=2 l=   9 prim:   OBJECT            :rsaEncryption
                       17:d=2  hl=2 l=   0 prim:   NULL
                       19:d=1  hl=4 l= 271 prim:  BIT STRING */
                    $tag = ord($this->_string_shift($key)); // skip over the BIT STRING / OCTET STRING tag
                    $this->_decodeLength($key); // skip over the BIT STRING / OCTET STRING length
                    // "The initial octet shall encode, as an unsigned binary integer wtih bit 1 as the least significant bit, the number of
                    //  unused bits in the final subsequent octet. The number shall be in the range zero to seven."
                    //  -- http://www.itu.int/ITU-T/studygroups/com17/languages/X.690-0207.pdf (section 8.6.2.2)
                    if (CRYPT_RSA_ASN1_BITSTRING == $tag) {
                        $this->_string_shift($key);
                    }
                    if (CRYPT_RSA_ASN1_SEQUENCE != ord($this->_string_shift($key))) {
                        return false;
                    }
                    if ($this->_decodeLength($key) != strlen((string) $key)) {
                        return false;
                    }
                    $tag = ord($this->_string_shift($key));
                }
                if (CRYPT_RSA_ASN1_INTEGER != $tag) {
                    return false;
                }

                $length = $this->_decodeLength($key);
                $temp   = $this->_string_shift($key, $length);
                if (1 != strlen($temp) || ord($temp) > 2) {
                    $components['modulus'] = new Math_BigInteger($temp, 256);
                    $this->_string_shift($key); // skip over CRYPT_RSA_ASN1_INTEGER
                    $length                                                                                    = $this->_decodeLength($key);
                    $components[CRYPT_RSA_PUBLIC_FORMAT_PKCS1 == $type ? 'publicExponent' : 'privateExponent'] = new Math_BigInteger(
                        $this->_string_shift($key, $length),
                        256
                    );

                    return $components;
                }
                if (CRYPT_RSA_ASN1_INTEGER != ord($this->_string_shift($key))) {
                    return false;
                }
                $length                = $this->_decodeLength($key);
                $components['modulus'] = new Math_BigInteger($this->_string_shift($key, $length), 256);
                $this->_string_shift($key);
                $length                       = $this->_decodeLength($key);
                $components['publicExponent'] = new Math_BigInteger($this->_string_shift($key, $length), 256);
                $this->_string_shift($key);
                $length                        = $this->_decodeLength($key);
                $components['privateExponent'] = new Math_BigInteger($this->_string_shift($key, $length), 256);
                $this->_string_shift($key);
                $length               = $this->_decodeLength($key);
                $components['primes'] = [
                    1 => new Math_BigInteger($this->_string_shift($key, $length), 256),
                ];
                $this->_string_shift($key);
                $length                 = $this->_decodeLength($key);
                $components['primes'][] = new Math_BigInteger($this->_string_shift($key, $length), 256);
                $this->_string_shift($key);
                $length                  = $this->_decodeLength($key);
                $components['exponents'] = [
                    1 => new Math_BigInteger($this->_string_shift($key, $length), 256),
                ];
                $this->_string_shift($key);
                $length                    = $this->_decodeLength($key);
                $components['exponents'][] = new Math_BigInteger($this->_string_shift($key, $length), 256);
                $this->_string_shift($key);
                $length                     = $this->_decodeLength($key);
                $components['coefficients'] = [
                    2 => new Math_BigInteger($this->_string_shift($key, $length), 256),
                ];

                if (! empty($key)) {
                    if (CRYPT_RSA_ASN1_SEQUENCE != ord($this->_string_shift($key))) {
                        return false;
                    }
                    $this->_decodeLength($key);
                    while (! empty($key)) {
                        if (CRYPT_RSA_ASN1_SEQUENCE != ord($this->_string_shift($key))) {
                            return false;
                        }
                        $this->_decodeLength($key);
                        $key                    = substr((string) $key, 1);
                        $length                 = $this->_decodeLength($key);
                        $components['primes'][] = new Math_BigInteger($this->_string_shift($key, $length), 256);
                        $this->_string_shift($key);
                        $length                    = $this->_decodeLength($key);
                        $components['exponents'][] = new Math_BigInteger($this->_string_shift($key, $length), 256);
                        $this->_string_shift($key);
                        $length                       = $this->_decodeLength($key);
                        $components['coefficients'][] = new Math_BigInteger($this->_string_shift($key, $length), 256);
                    }
                }

                return $components;
            case CRYPT_RSA_PUBLIC_FORMAT_OPENSSH:
                $parts = explode(' ', $key, 3);

                $key = isset($parts[1]) ? base64_decode($parts[1]) : false;
                if (false === $key) {
                    return false;
                }

                $comment = $parts[2] ?? false;

                $cleanup = str_starts_with($key, "\0\0\0\7ssh-rsa");

                if (strlen($key) <= 4) {
                    return false;
                }
                $unpack = unpack('Nlength', $this->_string_shift($key, 4));
                extract($unpack);
                $publicExponent = new Math_BigInteger($this->_string_shift($key, $length), -256);
                if (strlen((string) $key) <= 4) {
                    return false;
                }
                $unpack2 = unpack('Nlength', $this->_string_shift($key, 4));
                extract($unpack2);
                $modulus = new Math_BigInteger($this->_string_shift($key, $length), -256);

                if ($cleanup && strlen((string) $key)) {
                    if (strlen((string) $key) <= 4) {
                        return false;
                    }
                    $unpack3 = unpack('Nlength', $this->_string_shift($key, 4));
                    extract($unpack3);
                    $realModulus = new Math_BigInteger($this->_string_shift($key, $length), -256);

                    return strlen((string) $key) ? false : [
                        'modulus'        => $realModulus,
                        'publicExponent' => $modulus,
                        'comment'        => $comment,
                    ];
                } else {
                    return strlen((string) $key) ? false : [
                        'modulus'        => $modulus,
                        'publicExponent' => $publicExponent,
                        'comment'        => $comment,
                    ];
                }
                // http://www.w3.org/TR/xmldsig-core/#sec-RSAKeyValue
                // http://en.wikipedia.org/wiki/XML_Signature
                // no break
            case CRYPT_RSA_PRIVATE_FORMAT_XML:
            case CRYPT_RSA_PUBLIC_FORMAT_XML:
                $this->components = [];

                $xml = xml_parser_create('UTF-8');
                xml_set_object($xml, $this);
                xml_set_element_handler($xml, '_start_element_handler', '_stop_element_handler');
                xml_set_character_data_handler($xml, '_data_handler');
                // add <xml></xml> to account for "dangling" tags like <BitStrength>...</BitStrength> that are sometimes added
                if (! xml_parse($xml, '<xml>' . $key . '</xml>')) {
                    return false;
                }

                return isset($this->components['modulus']) && isset($this->components['publicExponent']) ? $this->components : false;
                // from PuTTY's SSHPUBK.C
            case CRYPT_RSA_PRIVATE_FORMAT_PUTTY:
                $components = [];
                $key        = preg_split('#\r\n|\r|\n#', $key);
                $type       = trim(preg_replace('#PuTTY-User-Key-File-2: (.+)#', '$1', $key[0]));
                if ('ssh-rsa' != $type) {
                    return false;
                }
                $encryption = trim(preg_replace('#Encryption: (.+)#', '$1', $key[1]));
                $comment    = trim(preg_replace('#Comment: (.+)#', '$1', $key[2]));

                $publicLength = trim(preg_replace('#Public-Lines: (\d+)#', '$1', $key[3]));
                $public       = base64_decode(implode('', array_map('trim', array_slice($key, 4, $publicLength))));
                $public       = substr($public, 11);
                $unpack4      = unpack('Nlength', $this->_string_shift($public, 4));
                extract($unpack4);
                $components['publicExponent'] = new Math_BigInteger($this->_string_shift($public, $length), -256);
                $unpack5                      = unpack('Nlength', $this->_string_shift($public, 4));
                extract($unpack5);
                $components['modulus'] = new Math_BigInteger($this->_string_shift($public, $length), -256);

                $privateLength = trim(preg_replace('#Private-Lines: (\d+)#', '$1', $key[$publicLength + 4]));
                $private       = base64_decode(implode('', array_map('trim', array_slice($key, $publicLength + 5, $privateLength))));

                switch ($encryption) {
                    case 'aes256-cbc':
                        if (! class_exists('Crypt_AES')) {
                            include_once 'Crypt/AES.php';
                        }
                        $symkey   = '';
                        $sequence = 0;
                        while (strlen($symkey) < 32) {
                            $temp = pack('Na*', $sequence++, $this->password);
                            $symkey .= pack('H*', sha1($temp));
                        }
                        $symkey = substr($symkey, 0, 32);
                        $crypto = new Crypt_AES();
                }

                if ('none' != $encryption) {
                    $crypto->setKey($symkey);
                    $crypto->disablePadding();
                    $private = $crypto->decrypt($private);
                    if (false === $private) {
                        return false;
                    }
                }
                $unpack6 = unpack('Nlength', $this->_string_shift($private, 4));

                extract($unpack6);
                if (strlen((string) $private) < $length) {
                    return false;
                }
                $components['privateExponent'] = new Math_BigInteger($this->_string_shift($private, $length), -256);
                $unpack7                       = unpack('Nlength', $this->_string_shift($private, 4));
                extract($unpack7);
                if (strlen((string) $private) < $length) {
                    return false;
                }
                $components['primes'] = [
                    1 => new Math_BigInteger($this->_string_shift($private, $length), -256),
                ];
                $unpack8 = unpack('Nlength', $this->_string_shift($private, 4));
                extract($unpack8);
                if (strlen((string) $private) < $length) {
                    return false;
                }
                $components['primes'][] = new Math_BigInteger($this->_string_shift($private, $length), -256);

                $temp                      = $components['primes'][1]->subtract($this->one);
                $components['exponents']   = [
                    1 => $components['publicExponent']->modInverse($temp),
                ];
                $temp                      = $components['primes'][2]->subtract($this->one);
                $components['exponents'][] = $components['publicExponent']->modInverse($temp);
                $unpack9                   = unpack('Nlength', $this->_string_shift($private, 4));

                extract($unpack9);
                if (strlen((string) $private) < $length) {
                    return false;
                }
                $components['coefficients'] = [
                    2 => new Math_BigInteger($this->_string_shift($private, $length), -256),
                ];

                return $components;
        }
    }

    /**
     * Extract raw BER from Base64 encoding.
     *
     * @param string $str
     *
     * @return string
     */
    public function _extractBER($str)
    {
        /* X.509 certs are assumed to be base64 encoded but sometimes they'll have additional things in them
         * above and beyond the ceritificate.
         * ie. some may have the following preceding the -----BEGIN CERTIFICATE----- line:
         *
         * Bag Attributes
         *     localKeyID: 01 00 00 00
         * subject=/O=organization/OU=org unit/CN=common name
         * issuer=/O=organization/CN=common name
         */
        $temp = preg_replace('#.*?^-+[^-]+-+#ms', '', $str, 1);
        // remove the -----BEGIN CERTIFICATE----- and -----END CERTIFICATE----- stuff
        $temp = preg_replace('#-+[^-]+-+#', '', $temp);
        // remove new lines
        $temp = str_replace(["\r", "\n", ' '], '', $temp);
        $temp = preg_match('#^[a-zA-Z\d/+]*={0,2}$#', $temp) ? base64_decode($temp) : false;

        return false != $temp ? $temp : $str;
    }

    /**
     * String Shift.
     *
     * Inspired by array_shift
     *
     * @param string           $string
     * @param optional Integer $index
     *
     * @return string
     */
    public function _string_shift(&$string, $index = 1)
    {
        $substr = substr($string, 0, $index);
        $string = substr($string, $index);

        return $substr;
    }

    /**
     * DER-decode the length.
     *
     * DER supports lengths up to (2**8)**127, however, we'll only support lengths up to (2**8)**4.  See
     *
     * {@link http://itu.int/ITU-T/studygroups/com17/languages/X.690-0207.pdf#p=13 X.690 paragraph 8.1.3} for more information.
     *
     * @param string $string
     *
     * @return int
     */
    public function _decodeLength(&$string)
    {
        $length = ord($this->_string_shift($string));
        if ($length & 0x80) { // definite length, long form
            $length &= 0x7F;
            $temp           = $this->_string_shift($string, $length);
            [, $length]     = unpack('N', substr(str_pad($temp, 4, chr(0), \STR_PAD_LEFT), -4));
        }

        return $length;
    }

    /**
     * Generates the smallest and largest numbers requiring $bits bits.
     *
     * @param int $bits
     *
     * @return array
     */
    public function _generateMinMax($bits)
    {
        $bytes = $bits >> 3;
        $min   = str_repeat(chr(0), $bytes);
        $max   = str_repeat(chr(0xFF), $bytes);
        $msb   = $bits & 7;
        if ($msb) {
            $min = chr(1 << ($msb - 1)) . $min;
            $max = chr((1 << $msb) - 1) . $max;
        } else {
            $min[0] = chr(0x80);
        }

        return [
            'min' => new Math_BigInteger($min, 256),
            'max' => new Math_BigInteger($max, 256),
        ];
    }

    /**
     * Convert a private key to the appropriate format.
     *
     * @see setPrivateKeyFormat()
     *
     * @return string
     */
    public function _convertPrivateKey($n, $e, $d, $primes, $exponents, $coefficients)
    {
        $signed     = CRYPT_RSA_PRIVATE_FORMAT_XML != $this->privateKeyFormat;
        $num_primes = count($primes);
        $raw        = [
            'version'         => 2 == $num_primes ? chr(0) : chr(1), // two-prime vs. multi
            'modulus'         => $n->toBytes($signed),
            'publicExponent'  => $e->toBytes($signed),
            'privateExponent' => $d->toBytes($signed),
            'prime1'          => $primes[1]->toBytes($signed),
            'prime2'          => $primes[2]->toBytes($signed),
            'exponent1'       => $exponents[1]->toBytes($signed),
            'exponent2'       => $exponents[2]->toBytes($signed),
            'coefficient'     => $coefficients[2]->toBytes($signed),
        ];

        // if the format in question does not support multi-prime rsa and multi-prime rsa was used,
        // call _convertPublicKey() instead.
        switch ($this->privateKeyFormat) {
            case CRYPT_RSA_PRIVATE_FORMAT_XML:
                if (2 != $num_primes) {
                    return false;
                }

                return "<RSAKeyValue>\r\n" .
                '  <Modulus>' . base64_encode((string) $raw['modulus']) . "</Modulus>\r\n" .
                '  <Exponent>' . base64_encode((string) $raw['publicExponent']) . "</Exponent>\r\n" .
                '  <P>' . base64_encode((string) $raw['prime1']) . "</P>\r\n" .
                '  <Q>' . base64_encode((string) $raw['prime2']) . "</Q>\r\n" .
                '  <DP>' . base64_encode((string) $raw['exponent1']) . "</DP>\r\n" .
                '  <DQ>' . base64_encode((string) $raw['exponent2']) . "</DQ>\r\n" .
                '  <InverseQ>' . base64_encode((string) $raw['coefficient']) . "</InverseQ>\r\n" .
                '  <D>' . base64_encode((string) $raw['privateExponent']) . "</D>\r\n" .
                '</RSAKeyValue>';
                break;
            case CRYPT_RSA_PRIVATE_FORMAT_PUTTY:
                if (2 != $num_primes) {
                    return false;
                }
                $key        = "PuTTY-User-Key-File-2: ssh-rsa\r\nEncryption: ";
                $encryption = (! empty($this->password) || is_string($this->password)) ? 'aes256-cbc' : 'none';
                $key .= $encryption;
                $key .= "\r\nComment: " . $this->comment . "\r\n";
                $public = pack(
                    'Na*Na*Na*',
                    strlen('ssh-rsa'),
                    'ssh-rsa',
                    strlen((string) $raw['publicExponent']),
                    $raw['publicExponent'],
                    strlen((string) $raw['modulus']),
                    $raw['modulus']
                );
                $source = pack(
                    'Na*Na*Na*Na*',
                    strlen('ssh-rsa'),
                    'ssh-rsa',
                    strlen($encryption),
                    $encryption,
                    strlen($this->comment),
                    $this->comment,
                    strlen($public),
                    $public
                );
                $public = base64_encode($public);
                $key .= 'Public-Lines: ' . ((strlen($public) + 63) >> 6) . "\r\n";
                $key .= chunk_split($public, 64);
                $private = pack(
                    'Na*Na*Na*Na*',
                    strlen((string) $raw['privateExponent']),
                    $raw['privateExponent'],
                    strlen((string) $raw['prime1']),
                    $raw['prime1'],
                    strlen((string) $raw['prime2']),
                    $raw['prime2'],
                    strlen((string) $raw['coefficient']),
                    $raw['coefficient']
                );
                if (empty($this->password) && ! is_string($this->password)) {
                    $source .= pack('Na*', strlen($private), $private);
                    $hashkey = 'putty-private-key-file-mac-key';
                } else {
                    $private .= crypt_random_string(16 - (strlen($private) & 15));
                    $source  .= pack('Na*', strlen($private), $private);
                    if (! class_exists('Crypt_AES')) {
                        include_once 'Crypt/AES.php';
                    }
                    $sequence = 0;
                    $symkey   = '';
                    while (strlen($symkey) < 32) {
                        $temp = pack('Na*', $sequence++, $this->password);
                        $symkey .= pack('H*', sha1($temp));
                    }
                    $symkey = substr($symkey, 0, 32);
                    $crypto = new Crypt_AES();

                    $crypto->setKey($symkey);
                    $crypto->disablePadding();
                    $private = $crypto->encrypt($private);
                    $hashkey = 'putty-private-key-file-mac-key' . $this->password;
                }

                $private = base64_encode((string) $private);
                $key .= 'Private-Lines: ' . ((strlen($private) + 63) >> 6) . "\r\n";
                $key .= chunk_split($private, 64);
                if (! class_exists('Crypt_Hash')) {
                    include_once 'Crypt/Hash.php';
                }
                $hash = new Crypt_Hash('sha1');
                $hash->setKey(pack('H*', sha1($hashkey)));
                $key .= 'Private-MAC: ' . bin2hex($hash->hash($source)) . "\r\n";

                return $key;
            default: // eg. CRYPT_RSA_PRIVATE_FORMAT_PKCS1
                $components = [];
                foreach ($raw as $name => $value) {
                    $components[$name] = pack('Ca*a*', CRYPT_RSA_ASN1_INTEGER, $this->_encodeLength(strlen((string) $value)), $value);
                }

                $RSAPrivateKey = implode('', $components);

                if ($num_primes > 2) {
                    $OtherPrimeInfos = '';
                    for ($i = 3; $i <= $num_primes; ++$i) {
                        // OtherPrimeInfos ::= SEQUENCE SIZE(1..MAX) OF OtherPrimeInfo
                        //
                        // OtherPrimeInfo ::= SEQUENCE {
                        //     prime             INTEGER,  -- ri
                        //     exponent          INTEGER,  -- di
                        //     coefficient       INTEGER   -- ti
                        // }
                        $OtherPrimeInfo = pack(
                            'Ca*a*',
                            CRYPT_RSA_ASN1_INTEGER,
                            $this->_encodeLength(strlen((string) $primes[$i]->toBytes(true))),
                            $primes[$i]->toBytes(true)
                        );
                        $OtherPrimeInfo .= pack(
                            'Ca*a*',
                            CRYPT_RSA_ASN1_INTEGER,
                            $this->_encodeLength(strlen((string) $exponents[$i]->toBytes(true))),
                            $exponents[$i]->toBytes(true)
                        );
                        $OtherPrimeInfo .= pack(
                            'Ca*a*',
                            CRYPT_RSA_ASN1_INTEGER,
                            $this->_encodeLength(strlen((string) $coefficients[$i]->toBytes(true))),
                            $coefficients[$i]->toBytes(true)
                        );
                        $OtherPrimeInfos .= pack(
                            'Ca*a*',
                            CRYPT_RSA_ASN1_SEQUENCE,
                            $this->_encodeLength(strlen($OtherPrimeInfo)),
                            $OtherPrimeInfo
                        );
                    }
                    $RSAPrivateKey .= pack(
                        'Ca*a*',
                        CRYPT_RSA_ASN1_SEQUENCE,
                        $this->_encodeLength(strlen($OtherPrimeInfos)),
                        $OtherPrimeInfos
                    );
                }

                $RSAPrivateKey = pack('Ca*a*', CRYPT_RSA_ASN1_SEQUENCE, $this->_encodeLength(strlen($RSAPrivateKey)), $RSAPrivateKey);

                if (CRYPT_RSA_PRIVATE_FORMAT_PKCS8 == $this->privateKeyFormat) {
                    $rsaOID        = pack('H*', '300d06092a864886f70d0101010500'); // hex version of MA0GCSqGSIb3DQEBAQUA
                    $RSAPrivateKey = pack(
                        'Ca*a*Ca*a*',
                        CRYPT_RSA_ASN1_INTEGER,
                        "\01\00",
                        $rsaOID,
                        4,
                        $this->_encodeLength(strlen($RSAPrivateKey)),
                        $RSAPrivateKey
                    );
                    $RSAPrivateKey = pack('Ca*a*', CRYPT_RSA_ASN1_SEQUENCE, $this->_encodeLength(strlen($RSAPrivateKey)), $RSAPrivateKey);
                    if (! empty($this->password) || is_string($this->password)) {
                        $salt           = crypt_random_string(8);
                        $iterationCount = 2048;

                        if (! class_exists('Crypt_DES')) {
                            include_once 'Crypt/DES.php';
                        }
                        $crypto = new Crypt_DES();
                        $crypto->setPassword($this->password, 'pbkdf1', 'md5', $salt, $iterationCount);
                        $RSAPrivateKey = $crypto->encrypt($RSAPrivateKey);

                        $parameters = pack(
                            'Ca*a*Ca*N',
                            CRYPT_RSA_ASN1_OCTETSTRING,
                            $this->_encodeLength(strlen($salt)),
                            $salt,
                            CRYPT_RSA_ASN1_INTEGER,
                            $this->_encodeLength(4),
                            $iterationCount
                        );
                        $pbeWithMD5AndDES_CBC = "\x2a\x86\x48\x86\xf7\x0d\x01\x05\x03";

                        $encryptionAlgorithm = pack(
                            'Ca*a*Ca*a*',
                            CRYPT_RSA_ASN1_OBJECT,
                            $this->_encodeLength(strlen($pbeWithMD5AndDES_CBC)),
                            $pbeWithMD5AndDES_CBC,
                            CRYPT_RSA_ASN1_SEQUENCE,
                            $this->_encodeLength(strlen($parameters)),
                            $parameters
                        );

                        $RSAPrivateKey = pack(
                            'Ca*a*Ca*a*',
                            CRYPT_RSA_ASN1_SEQUENCE,
                            $this->_encodeLength(strlen($encryptionAlgorithm)),
                            $encryptionAlgorithm,
                            CRYPT_RSA_ASN1_OCTETSTRING,
                            $this->_encodeLength(strlen((string) $RSAPrivateKey)),
                            $RSAPrivateKey
                        );

                        $RSAPrivateKey = pack(
                            'Ca*a*',
                            CRYPT_RSA_ASN1_SEQUENCE,
                            $this->_encodeLength(strlen($RSAPrivateKey)),
                            $RSAPrivateKey
                        );

                        $RSAPrivateKey = "-----BEGIN ENCRYPTED PRIVATE KEY-----\r\n" .
                            chunk_split(base64_encode($RSAPrivateKey), 64) .
                            '-----END ENCRYPTED PRIVATE KEY-----';
                    } else {
                        $RSAPrivateKey = "-----BEGIN PRIVATE KEY-----\r\n" .
                            chunk_split(base64_encode($RSAPrivateKey), 64) .
                            '-----END PRIVATE KEY-----';
                    }

                    return $RSAPrivateKey;
                }

                if (! empty($this->password) || is_string($this->password)) {
                    $iv     = crypt_random_string(8);
                    $symkey = pack('H*', md5($this->password . $iv)); // symkey is short for symmetric key
                    $symkey .= substr(pack('H*', md5($symkey . $this->password . $iv)), 0, 8);
                    if (! class_exists('Crypt_TripleDES')) {
                        include_once 'Crypt/TripleDES.php';
                    }
                    $des = new Crypt_TripleDES();
                    $des->setKey($symkey);
                    $des->setIV($iv);
                    $iv            = strtoupper(bin2hex($iv));
                    $RSAPrivateKey = "-----BEGIN RSA PRIVATE KEY-----\r\n" .
                        "Proc-Type: 4,ENCRYPTED\r\n" .
                        "DEK-Info: DES-EDE3-CBC,$iv\r\n" .
                        "\r\n" .
                        chunk_split(base64_encode((string) $des->encrypt($RSAPrivateKey)), 64) .
                        '-----END RSA PRIVATE KEY-----';
                } else {
                    $RSAPrivateKey = "-----BEGIN RSA PRIVATE KEY-----\r\n" .
                        chunk_split(base64_encode($RSAPrivateKey), 64) .
                        '-----END RSA PRIVATE KEY-----';
                }

                return $RSAPrivateKey;
        }
    }

    /**
     * DER-encode the length.
     *
     * DER supports lengths up to (2**8)**127, however, we'll only support lengths up to (2**8)**4.  See
     *
     * {@link http://itu.int/ITU-T/studygroups/com17/languages/X.690-0207.pdf#p=13 X.690 paragraph 8.1.3} for more information.
     *
     * @param int $length
     *
     * @return string
     */
    public function _encodeLength($length)
    {
        if ($length <= 0x7F) {
            return chr($length);
        }

        $temp = ltrim(pack('N', $length), chr(0));

        return pack('Ca*', 0x80 | strlen($temp), $temp);
    }

    /**
     * Convert a public key to the appropriate format.
     *
     * @see setPublicKeyFormat()
     *
     * @return string
     */
    public function _convertPublicKey($n, $e)
    {
        $signed = CRYPT_RSA_PUBLIC_FORMAT_XML != $this->publicKeyFormat;

        $modulus        = $n->toBytes($signed);
        $publicExponent = $e->toBytes($signed);

        switch ($this->publicKeyFormat) {
            case CRYPT_RSA_PUBLIC_FORMAT_RAW:
                return [
                    'e' => $e->copy(),
                    'n' => $n->copy(),
                ];
            case CRYPT_RSA_PUBLIC_FORMAT_XML:
                return "<RSAKeyValue>\r\n" .
                '  <Modulus>' . base64_encode((string) $modulus) . "</Modulus>\r\n" .
                '  <Exponent>' . base64_encode((string) $publicExponent) . "</Exponent>\r\n" .
                '</RSAKeyValue>';
                break;
            case CRYPT_RSA_PUBLIC_FORMAT_OPENSSH:
                // from <http://tools.ietf.org/html/rfc4253#page-15>:
                // string    "ssh-rsa"
                // mpint     e
                // mpint     n
                $RSAPublicKey = pack(
                    'Na*Na*Na*',
                    strlen('ssh-rsa'),
                    'ssh-rsa',
                    strlen((string) $publicExponent),
                    $publicExponent,
                    strlen((string) $modulus),
                    $modulus
                );
                $RSAPublicKey = 'ssh-rsa ' . base64_encode($RSAPublicKey) . ' ' . $this->comment;

                return $RSAPublicKey;
            default: // eg. CRYPT_RSA_PUBLIC_FORMAT_PKCS1_RAW or CRYPT_RSA_PUBLIC_FORMAT_PKCS1
                // from <http://tools.ietf.org/html/rfc3447#appendix-A.1.1>:
                // RSAPublicKey ::= SEQUENCE {
                //     modulus           INTEGER,  -- n
                //     publicExponent    INTEGER   -- e
                // }
                $components = [
                    'modulus'        => pack('Ca*a*', CRYPT_RSA_ASN1_INTEGER, $this->_encodeLength(strlen((string) $modulus)), $modulus),
                    'publicExponent' => pack(
                        'Ca*a*',
                        CRYPT_RSA_ASN1_INTEGER,
                        $this->_encodeLength(strlen((string) $publicExponent)),
                        $publicExponent
                    ),
                ];

                $RSAPublicKey = pack(
                    'Ca*a*a*',
                    CRYPT_RSA_ASN1_SEQUENCE,
                    $this->_encodeLength(strlen($components['modulus']) + strlen($components['publicExponent'])),
                    $components['modulus'],
                    $components['publicExponent']
                );

                if (CRYPT_RSA_PUBLIC_FORMAT_PKCS1_RAW == $this->publicKeyFormat) {
                    $RSAPublicKey = "-----BEGIN RSA PUBLIC KEY-----\r\n" .
                        chunk_split(base64_encode($RSAPublicKey), 64) .
                        '-----END RSA PUBLIC KEY-----';
                } else {
                    // sequence(oid(1.2.840.113549.1.1.1), null)) = rsaEncryption.
                    $rsaOID       = pack('H*', '300d06092a864886f70d0101010500'); // hex version of MA0GCSqGSIb3DQEBAQUA
                    $RSAPublicKey = chr(0) . $RSAPublicKey;
                    $RSAPublicKey = chr(3) . $this->_encodeLength(strlen($RSAPublicKey)) . $RSAPublicKey;

                    $RSAPublicKey = pack(
                        'Ca*a*',
                        CRYPT_RSA_ASN1_SEQUENCE,
                        $this->_encodeLength(strlen($rsaOID . $RSAPublicKey)),
                        $rsaOID . $RSAPublicKey
                    );

                    $RSAPublicKey = "-----BEGIN PUBLIC KEY-----\r\n" .
                        chunk_split(base64_encode($RSAPublicKey), 64) .
                        '-----END PUBLIC KEY-----';
                }

                return $RSAPublicKey;
        }
    }

    /**
     * Returns the key size.
     *
     * More specifically, this returns the size of the modulo in bits.
     *
     * @return int
     */
    public function getSize()
    {
        return ! isset($this->modulus) ? 0 : strlen($this->modulus->toBits());
    }

    /**
     * Start Element Handler.
     *
     * Called by xml_set_element_handler()
     *
     * @param resource $parser
     * @param string   $name
     * @param array    $attribs
     */
    public function _start_element_handler($parser, $name, $attribs)
    {
        //$name = strtoupper($name);
        switch ($name) {
            case 'MODULUS':
                $this->current = &$this->components['modulus'];
                break;
            case 'EXPONENT':
                $this->current = &$this->components['publicExponent'];
                break;
            case 'P':
                $this->current = &$this->components['primes'][1];
                break;
            case 'Q':
                $this->current = &$this->components['primes'][2];
                break;
            case 'DP':
                $this->current = &$this->components['exponents'][1];
                break;
            case 'DQ':
                $this->current = &$this->components['exponents'][2];
                break;
            case 'INVERSEQ':
                $this->current = &$this->components['coefficients'][2];
                break;
            case 'D':
                $this->current = &$this->components['privateExponent'];
        }
        $this->current = '';
    }

    /**
     * Stop Element Handler.
     *
     * Called by xml_set_element_handler()
     *
     * @param resource $parser
     * @param string   $name
     */
    public function _stop_element_handler($parser, $name)
    {
        if (isset($this->current)) {
            $this->current = new Math_BigInteger(base64_decode((string) $this->current), 256);
            unset($this->current);
        }
    }

    /**
     * Data Handler.
     *
     * Called by xml_set_character_data_handler()
     *
     * @param resource $parser
     * @param string   $data
     */
    public function _data_handler($parser, $data)
    {
        if (! isset($this->current) || is_object($this->current)) {
            return;
        }
        $this->current .= trim($data);
    }

    /**
     * Sets the password.
     *
     * Private keys can be encrypted with a password.  To unset the password, pass in the empty string or false. Or
     * rather, pass in $password such that empty($password) && !is_string($password) is true.
     *
     * @see createKey()
     * @see loadKey()
     *
     * @param string $password
     */
    public function setPassword($password = false)
    {
        $this->password = $password;
    }

    /**
     * Defines the private key.
     *
     * If phpseclib guessed a private key was a public key and loaded it as such it might be desirable to force
     * phpseclib to treat the key as a private key. This function will do that.
     *
     * Do note that when a new key is loaded the index will be cleared.
     *
     * Returns true on success, false on failure
     *
     * @see getPublicKey()
     *
     * @param string $key  optional
     * @param int    $type optional
     *
     * @return bool
     */
    public function setPrivateKey($key = false, $type = false)
    {
        if (false === $key && ! empty($this->publicExponent)) {
            unset($this->publicExponent);

            return true;
        }

        $rsa = new self();
        if (! $rsa->loadKey($key, $type)) {
            return false;
        }
        unset($rsa->publicExponent);

        // don't overwrite the old key if the new key is invalid
        $this->loadKey($rsa);

        return true;
    }

    /**
     * Loads a public or private key.
     *
     * Returns true on success and false on failure (ie. an incorrect password was provided or the key was malformed)
     *
     * @param string $key
     * @param int    $type optional
     */
    public function loadKey($key, $type = false)
    {
        if (is_object($key) && 'crypt_rsa' == strtolower((string) $key::class)) {
            $this->privateKeyFormat = $key->privateKeyFormat;
            $this->publicKeyFormat  = $key->publicKeyFormat;
            $this->k                = $key->k;
            $this->hLen             = $key->hLen;
            $this->sLen             = $key->sLen;
            $this->mgfHLen          = $key->mgfHLen;
            $this->encryptionMode   = $key->encryptionMode;
            $this->signatureMode    = $key->signatureMode;
            $this->password         = $key->password;
            $this->configFile       = $key->configFile;
            $this->comment          = $key->comment;

            if (is_object($key->hash)) {
                $this->hash = new Crypt_Hash($key->hash->getHash());
            }
            if (is_object($key->mgfHash)) {
                $this->mgfHash = new Crypt_Hash($key->mgfHash->getHash());
            }

            if (is_object($key->modulus)) {
                $this->modulus = $key->modulus->copy();
            }
            if (is_object($key->exponent)) {
                $this->exponent = $key->exponent->copy();
            }
            if (is_object($key->publicExponent)) {
                $this->publicExponent = $key->publicExponent->copy();
            }

            $this->primes       = [];
            $this->exponents    = [];
            $this->coefficients = [];

            foreach ($this->primes as $prime) {
                $this->primes[] = $prime->copy();
            }
            foreach ($this->exponents as $exponent) {
                $this->exponents[] = $exponent->copy();
            }
            foreach ($this->coefficients as $coefficient) {
                $this->coefficients[] = $coefficient->copy();
            }

            return true;
        }

        if (false === $type) {
            $types = [
                CRYPT_RSA_PUBLIC_FORMAT_RAW,
                CRYPT_RSA_PRIVATE_FORMAT_PKCS1,
                CRYPT_RSA_PRIVATE_FORMAT_XML,
                CRYPT_RSA_PRIVATE_FORMAT_PUTTY,
                CRYPT_RSA_PUBLIC_FORMAT_OPENSSH,
            ];
            foreach ($types as $type) {
                $components = $this->_parseKey($key, $type);
                if (false !== $components) {
                    break;
                }
            }
        } else {
            $components = $this->_parseKey($key, $type);
        }

        if (false === $components) {
            return false;
        }

        if (isset($components['comment']) && false !== $components['comment']) {
            $this->comment = $components['comment'];
        }
        $this->modulus  = $components['modulus'];
        $this->k        = strlen((string) $this->modulus->toBytes());
        $this->exponent = $components['privateExponent'] ?? $components['publicExponent'];
        if (isset($components['primes'])) {
            $this->primes         = $components['primes'];
            $this->exponents      = $components['exponents'];
            $this->coefficients   = $components['coefficients'];
            $this->publicExponent = $components['publicExponent'];
        } else {
            $this->primes         = [];
            $this->exponents      = [];
            $this->coefficients   = [];
            $this->publicExponent = false;
        }

        switch ($type) {
            case CRYPT_RSA_PUBLIC_FORMAT_OPENSSH:
            case CRYPT_RSA_PUBLIC_FORMAT_RAW:
                $this->setPublicKey();
                break;
            case CRYPT_RSA_PRIVATE_FORMAT_PKCS1:
                switch (true) {
                    case str_contains($key, '-BEGIN PUBLIC KEY-'):
                    case str_contains($key, '-BEGIN RSA PUBLIC KEY-'):
                        $this->setPublicKey();
                }
        }

        return true;
    }

    /**
     * Defines the public key.
     *
     * Some private key formats define the public exponent and some don't.  Those that don't define it are problematic
     * when used in certain contexts.  For example, in SSH-2, RSA authentication works by sending the public key along
     * with a message signed by the private key to the server.  The SSH-2 server looks the public key up in an index of
     * public keys and if it's present then proceeds to verify the signature.  Problem is, if your private key doesn't
     * include the public exponent this won't work unless you manually add the public exponent. phpseclib tries to guess
     * if the key being used is the public key but in the event that it guesses incorrectly you might still want to
     * explicitly set the key as being public.
     *
     * Do note that when a new key is loaded the index will be cleared.
     *
     * Returns true on success, false on failure
     *
     * @see getPublicKey()
     *
     * @param string $key  optional
     * @param int    $type optional
     *
     * @return bool
     */
    public function setPublicKey($key = false, $type = false)
    {
        // if a public key has already been loaded return false
        if (! empty($this->publicExponent)) {
            return false;
        }

        if (false === $key && ! empty($this->modulus)) {
            $this->publicExponent = $this->exponent;

            return true;
        }

        if (false === $type) {
            $types = [
                CRYPT_RSA_PUBLIC_FORMAT_RAW,
                CRYPT_RSA_PUBLIC_FORMAT_PKCS1,
                CRYPT_RSA_PUBLIC_FORMAT_XML,
                CRYPT_RSA_PUBLIC_FORMAT_OPENSSH,
            ];
            foreach ($types as $type) {
                $components = $this->_parseKey($key, $type);
                if (false !== $components) {
                    break;
                }
            }
        } else {
            $components = $this->_parseKey($key, $type);
        }

        if (false === $components) {
            return false;
        }

        if (empty($this->modulus) || ! $this->modulus->equals($components['modulus'])) {
            $this->modulus  = $components['modulus'];
            $this->exponent = $this->publicExponent = $components['publicExponent'];

            return true;
        }

        $this->publicExponent = $components['publicExponent'];

        return true;
    }

    /**
     * Returns the public key.
     *
     * The public key is only returned under two circumstances - if the private key had the public key embedded within
     * it or if the public key was set via setPublicKey().  If the currently loaded key is supposed to be the public key
     * this function won't return it since this library, for the most part, doesn't distinguish between public and
     * private keys.
     *
     * @see getPublicKey()
     *
     * @param int $type optional
     */
    public function getPublicKey($type = CRYPT_RSA_PUBLIC_FORMAT_PKCS8)
    {
        if (empty($this->modulus) || empty($this->publicExponent)) {
            return false;
        }

        $oldFormat             = $this->publicKeyFormat;
        $this->publicKeyFormat = $type;
        $temp                  = $this->_convertPublicKey($this->modulus, $this->publicExponent);
        $this->publicKeyFormat = $oldFormat;

        return $temp;
    }

    /**
     * __toString() magic method.
     */
    public function __toString(): string
    {
        $key = $this->getPrivateKey($this->privateKeyFormat);
        if (false !== $key) {
            return (string) $key;
        }
        $key = $this->_getPrivatePublicKey($this->publicKeyFormat);

        return (string) (false !== $key ? $key : '');
    }

    /**
     * Returns the private key.
     *
     * The private key is only returned if the currently loaded key contains the constituent prime numbers.
     *
     * @see getPublicKey()
     *
     * @param int $type optional
     */
    public function getPrivateKey($type = CRYPT_RSA_PUBLIC_FORMAT_PKCS1)
    {
        if (empty($this->primes)) {
            return false;
        }

        $oldFormat              = $this->privateKeyFormat;
        $this->privateKeyFormat = $type;
        $temp                   = $this->_convertPrivateKey($this->modulus, $this->publicExponent, $this->exponent, $this->primes, $this->exponents, $this->coefficients);
        $this->privateKeyFormat = $oldFormat;

        return $temp;
    }

    /**
     * Returns a minimalistic private key.
     *
     * Returns the private key without the prime number constituants.  Structurally identical to a public key that
     * hasn't been set as the public key
     *
     * @see getPrivateKey()
     */
    public function _getPrivatePublicKey($mode = CRYPT_RSA_PUBLIC_FORMAT_PKCS8)
    {
        if (empty($this->modulus) || empty($this->exponent)) {
            return false;
        }

        $oldFormat             = $this->publicKeyFormat;
        $this->publicKeyFormat = $mode;
        $temp                  = $this->_convertPublicKey($this->modulus, $this->exponent);
        $this->publicKeyFormat = $oldFormat;

        return $temp;
    }

    /**
     * __clone() magic method.
     */
    public function __clone()
    {
        $key = new self();
        $key->loadKey($this);

        return $key;
    }

    /**
     * Determines the private key format.
     *
     * @see createKey()
     *
     * @param int $format
     */
    public function setPrivateKeyFormat($format)
    {
        $this->privateKeyFormat = $format;
    }

    /**
     * Determines the public key format.
     *
     * @see createKey()
     *
     * @param int $format
     */
    public function setPublicKeyFormat($format)
    {
        $this->publicKeyFormat = $format;
    }

    /**
     * Determines which hashing function should be used.
     *
     * Used with signature production / verification and (if the encryption mode is CRYPT_RSA_ENCRYPTION_OAEP)
     * encryption and decryption.  If $hash isn't supported, sha1 is used.
     *
     * @param string $hash
     */
    public function setHash($hash)
    {
        // Crypt_Hash supports algorithms that PKCS#1 doesn't support.  md5-96 and sha1-96, for example.
        switch ($hash) {
            case 'md2':
            case 'md5':
            case 'sha1':
            case 'sha256':
            case 'sha384':
            case 'sha512':
                $this->hash     = new Crypt_Hash($hash);
                $this->hashName = $hash;
                break;
            default:
                $this->hash     = new Crypt_Hash('sha1');
                $this->hashName = 'sha1';
        }
        $this->hLen = $this->hash->getLength();
    }

    /**
     * Determines which hashing function should be used for the mask generation function.
     *
     * The mask generation function is used by CRYPT_RSA_ENCRYPTION_OAEP and CRYPT_RSA_SIGNATURE_PSS and although it's
     * best if Hash and MGFHash are set to the same thing this is not a requirement.
     *
     * @param string $hash
     */
    public function setMGFHash($hash)
    {
        // Crypt_Hash supports algorithms that PKCS#1 doesn't support.  md5-96 and sha1-96, for example.
        $this->mgfHash = match ($hash) {
            'md2', 'md5', 'sha1', 'sha256', 'sha384', 'sha512' => new Crypt_Hash($hash),
            default => new Crypt_Hash('sha1'),
        };
        $this->mgfHLen = $this->mgfHash->getLength();
    }

    /**
     * Determines the salt length.
     *
     * To quote from {@link http://tools.ietf.org/html/rfc3447#page-38 RFC3447#page-38}:
     *
     * Typical salt lengths in octets are hLen (the length of the output of the hash function Hash) and 0.
     */
    public function setSaltLength($sLen)
    {
        $this->sLen = $sLen;
    }

    /**
     * RSAES-OAEP-DECRYPT.
     *
     * See {@link http://tools.ietf.org/html/rfc3447#section-7.1.2 RFC3447#section-7.1.2}.  The fact that the error
     * messages aren't distinguishable from one another hinders debugging, but, to quote from RFC3447#section-7.1.2:
     *
     * Note.  Care must be taken to ensure that an opponent cannot distinguish the different error conditions in Step
     * 3.g, whether by error message or timing, or, more generally, learn partial information about the encoded message
     * EM.  Otherwise an opponent may be able to obtain useful information about the decryption of the ciphertext C,
     * leading to a chosen-ciphertext attack such as the one observed by Manger [36].
     *
     * As for $l...  to quote from {@link http://tools.ietf.org/html/rfc3447#page-17 RFC3447#page-17}:
     *
     * Both the encryption and the decryption operations of RSAES-OAEP take the value of a label L as input.  In this
     * version of PKCS #1, L is the empty string; other uses of the label are outside the scope of this document.
     *
     * @param string $c
     * @param string $l
     *
     * @return string
     */
    public function _rsaes_oaep_decrypt($c, $l = '')
    {
        // Length checking

        // if $l is larger than two million terrabytes and you're using sha1, PKCS#1 suggests a "Label too long" error
        // be output.

        if (strlen($c) != $this->k || $this->k < 2 * $this->hLen + 2) {
            trigger_error('Decryption error');

            return false;
        }

        // RSA decryption

        $c = $this->_os2ip($c);
        $m = $this->_rsadp($c);
        if (false === $m) {
            trigger_error('Decryption error');

            return false;
        }
        $em = $this->_i2osp($m, $this->k);

        // EME-OAEP decoding

        $lHash      = $this->hash->hash($l);
        $y          = ord($em[0]);
        $maskedSeed = substr($em, 1, $this->hLen);
        $maskedDB   = substr($em, $this->hLen + 1);
        $seedMask   = $this->_mgf1($maskedDB, $this->hLen);
        $seed       = $maskedSeed ^ $seedMask;
        $dbMask     = $this->_mgf1($seed, $this->k - $this->hLen - 1);
        $db         = $maskedDB ^ $dbMask;
        $lHash2     = substr($db, 0, $this->hLen);
        $m          = substr($db, $this->hLen);
        if ($lHash != $lHash2) {
            trigger_error('Decryption error');

            return false;
        }
        $m = ltrim($m, chr(0));
        if (1 != ord($m[0])) {
            trigger_error('Decryption error');

            return false;
        }

        // Output the message M

        return substr($m, 1);
    }

    /**
     * Octet-String-to-Integer primitive.
     *
     * See {@link http://tools.ietf.org/html/rfc3447#section-4.2 RFC3447#section-4.2}.
     *
     * @param string $x
     *
     * @return Math_BigInteger
     */
    public function _os2ip($x)
    {
        return new Math_BigInteger($x, 256);
    }

    /**
     * RSADP.
     *
     * See {@link http://tools.ietf.org/html/rfc3447#section-5.1.2 RFC3447#section-5.1.2}.
     *
     * @param Math_BigInteger $c
     *
     * @return Math_BigInteger
     */
    public function _rsadp($c)
    {
        if ($c->compare($this->zero) < 0 || $c->compare($this->modulus) > 0) {
            trigger_error('Ciphertext representative out of range');

            return false;
        }

        return $this->_exponentiate($c);
    }

    /**
     * Exponentiate with or without Chinese Remainder Theorem.
     *
     * See {@link http://tools.ietf.org/html/rfc3447#section-5.1.1 RFC3447#section-5.1.2}.
     *
     * @param Math_BigInteger $x
     *
     * @return Math_BigInteger
     */
    public function _exponentiate($x)
    {
        if (empty($this->primes) || empty($this->coefficients) || empty($this->exponents)) {
            return $x->modPow($this->exponent, $this->modulus);
        }

        $num_primes = count($this->primes);

        if (defined('CRYPT_RSA_DISABLE_BLINDING')) {
            $m_i = [
                1 => $x->modPow($this->exponents[1], $this->primes[1]),
                2 => $x->modPow($this->exponents[2], $this->primes[2]),
            ];
            $h         = $m_i[1]->subtract($m_i[2]);
            $h         = $h->multiply($this->coefficients[2]);
            [, $h]     = $h->divide($this->primes[1]);
            $m         = $m_i[2]->add($h->multiply($this->primes[2]));

            $r = $this->primes[1];
            for ($i = 3; $i <= $num_primes; ++$i) {
                $m_i = $x->modPow($this->exponents[$i], $this->primes[$i]);

                $r = $r->multiply($this->primes[$i - 1]);

                $h         = $m_i->subtract($m);
                $h         = $h->multiply($this->coefficients[$i]);
                [, $h]     = $h->divide($this->primes[$i]);

                $m = $m->add($r->multiply($h));
            }
        } else {
            $smallest = $this->primes[1];
            for ($i = 2; $i <= $num_primes; ++$i) {
                if ($smallest->compare($this->primes[$i]) > 0) {
                    $smallest = $this->primes[$i];
                }
            }

            $one = new Math_BigInteger(1);

            $r = $one->random($one, $smallest->subtract($one));

            $m_i = [
                1 => $this->_blind($x, $r, 1),
                2 => $this->_blind($x, $r, 2),
            ];
            $h         = $m_i[1]->subtract($m_i[2]);
            $h         = $h->multiply($this->coefficients[2]);
            [, $h]     = $h->divide($this->primes[1]);
            $m         = $m_i[2]->add($h->multiply($this->primes[2]));

            $r = $this->primes[1];
            for ($i = 3; $i <= $num_primes; ++$i) {
                $m_i = $this->_blind($x, $r, $i);

                $r = $r->multiply($this->primes[$i - 1]);

                $h         = $m_i->subtract($m);
                $h         = $h->multiply($this->coefficients[$i]);
                [, $h]     = $h->divide($this->primes[$i]);

                $m = $m->add($r->multiply($h));
            }
        }

        return $m;
    }

    /**
     * Performs RSA Blinding.
     *
     * Protects against timing attacks by employing RSA Blinding. Returns $x->modPow($this->exponents[$i],
     * $this->primes[$i])
     *
     * @param Math_BigInteger $x
     * @param Math_BigInteger $r
     * @param int             $i
     *
     * @return Math_BigInteger
     */
    public function _blind($x, $r, $i)
    {
        $x = $x->multiply($r->modPow($this->publicExponent, $this->primes[$i]));
        $x = $x->modPow($this->exponents[$i], $this->primes[$i]);

        $r         = $r->modInverse($this->primes[$i]);
        $x         = $x->multiply($r);
        [, $x]     = $x->divide($this->primes[$i]);

        return $x;
    }

    /**
     * Integer-to-Octet-String primitive.
     *
     * See {@link http://tools.ietf.org/html/rfc3447#section-4.1 RFC3447#section-4.1}.
     *
     * @param Math_BigInteger $x
     * @param int             $xLen
     *
     * @return string
     */
    public function _i2osp($x, $xLen)
    {
        $x = $x->toBytes();
        if (strlen($x) > $xLen) {
            trigger_error('Integer too large');

            return false;
        }

        return str_pad($x, $xLen, chr(0), \STR_PAD_LEFT);
    }

    /**
     * MGF1.
     *
     * See {@link http://tools.ietf.org/html/rfc3447#appendix-B.2.1 RFC3447#appendix-B.2.1}.
     *
     * @param string $mgfSeed
     *
     * @return string
     */
    public function _mgf1($mgfSeed, $maskLen)
    {
        // if $maskLen would yield strings larger than 4GB, PKCS#1 suggests a "Mask too long" error be output.

        $t     = '';
        $count = ceil($maskLen / $this->mgfHLen);
        for ($i = 0; $i < $count; ++$i) {
            $c = pack('N', $i);
            $t .= $this->mgfHash->hash($mgfSeed . $c);
        }

        return substr($t, 0, $maskLen);
    }

    /**
     * RSAES-PKCS1-V1_5-DECRYPT.
     *
     * See {@link http://tools.ietf.org/html/rfc3447#section-7.2.2 RFC3447#section-7.2.2}.
     *
     * For compatibility purposes, this function departs slightly from the description given in RFC3447. The reason
     * being that RFC2313#section-8.1 (PKCS#1 v1.5) states that ciphertext's encrypted by the private key should have
     * the second byte set to either 0 or 1 and that ciphertext's encrypted by the public key should have the second
     * byte set to 2.  In RFC3447 (PKCS#1 v2.1), the second byte is supposed to be 2 regardless of which key is used.
     * For compatibility purposes, we'll just check to make sure the second byte is 2 or less.  If it is, we'll accept
     * the decrypted string as valid.
     *
     * As a consequence of this, a private key encrypted ciphertext produced with Crypt_RSA may not decrypt with a
     * strictly PKCS#1 v1.5 compliant RSA implementation.  Public key encrypted ciphertext's should but not private key
     * encrypted ciphertext's.
     *
     * @param string $c
     *
     * @return string
     */
    public function _rsaes_pkcs1_v1_5_decrypt($c)
    {
        // Length checking

        if (strlen($c) != $this->k) { // or if k < 11
            trigger_error('Decryption error');

            return false;
        }

        // RSA decryption

        $c = $this->_os2ip($c);
        $m = $this->_rsadp($c);

        if (false === $m) {
            trigger_error('Decryption error');

            return false;
        }
        $em = $this->_i2osp($m, $this->k);

        // EME-PKCS1-v1_5 decoding

        if (0 != ord($em[0]) || ord($em[1]) > 2) {
            trigger_error('Decryption error');

            return false;
        }

        $ps = substr($em, 2, strpos($em, chr(0), 2) - 2);
        $m  = substr($em, strlen($ps) + 3);

        if (strlen($ps) < 8) {
            trigger_error('Decryption error');

            return false;
        }

        // Output M

        return $m;
    }

    /**
     * Set Encryption Mode.
     *
     * Valid values include CRYPT_RSA_ENCRYPTION_OAEP and CRYPT_RSA_ENCRYPTION_PKCS1.
     *
     * @param int $mode
     */
    public function setEncryptionMode($mode)
    {
        $this->encryptionMode = $mode;
    }

    /**
     * Set Signature Mode.
     *
     * Valid values include CRYPT_RSA_SIGNATURE_PSS and CRYPT_RSA_SIGNATURE_PKCS1
     *
     * @param int $mode
     */
    public function setSignatureMode($mode)
    {
        $this->signatureMode = $mode;
    }

    /**
     * Get public key comment.
     *
     * @return string
     */
    public function getComment()
    {
        return $this->comment;
    }

    /**
     * Set public key comment.
     *
     * @param string $comment
     */
    public function setComment($comment)
    {
        $this->comment = $comment;
    }

    /**
     * Encryption.
     *
     * Both CRYPT_RSA_ENCRYPTION_OAEP and CRYPT_RSA_ENCRYPTION_PKCS1 both place limits on how long $plaintext can be. If
     * $plaintext exceeds those limits it will be broken up so that it does and the resultant ciphertext's will be
     * concatenated together.
     *
     * @see decrypt()
     *
     * @param string $plaintext
     *
     * @return string
     */
    public function encrypt($plaintext)
    {
        switch ($this->encryptionMode) {
            case CRYPT_RSA_ENCRYPTION_PKCS1:
                $length = $this->k - 11;
                if ($length <= 0) {
                    return false;
                }

                $plaintext  = str_split($plaintext, $length);
                $ciphertext = '';
                foreach ($plaintext as $m) {
                    $ciphertext .= $this->_rsaes_pkcs1_v1_5_encrypt($m);
                }

                return $ciphertext;
                //case CRYPT_RSA_ENCRYPTION_OAEP:
            default:
                $length = $this->k - 2 * $this->hLen - 2;
                if ($length <= 0) {
                    return false;
                }

                $plaintext  = str_split($plaintext, $length);
                $ciphertext = '';
                foreach ($plaintext as $m) {
                    $ciphertext .= $this->_rsaes_oaep_encrypt($m);
                }

                return $ciphertext;
        }
    }

    /**
     * RSAES-PKCS1-V1_5-ENCRYPT.
     *
     * See {@link http://tools.ietf.org/html/rfc3447#section-7.2.1 RFC3447#section-7.2.1}.
     *
     * @param string $m
     *
     * @return string
     */
    public function _rsaes_pkcs1_v1_5_encrypt($m)
    {
        $mLen = strlen($m);

        // Length checking

        if ($mLen > $this->k - 11) {
            trigger_error('Message too long');

            return false;
        }

        // EME-PKCS1-v1_5 encoding

        $psLen = $this->k - $mLen - 3;
        $ps    = '';
        while (strlen($ps) != $psLen) {
            $temp = crypt_random_string($psLen - strlen($ps));
            $temp = str_replace("\x00", '', $temp);
            $ps .= $temp;
        }
        $type = 2;
        // see the comments of _rsaes_pkcs1_v1_5_decrypt() to understand why this is being done
        if (defined('CRYPT_RSA_PKCS15_COMPAT') && (! isset($this->publicExponent) || $this->exponent !== $this->publicExponent)) {
            $type = 1;
            // "The padding string PS shall consist of k-3-||D|| octets. ... for block type 01, they shall have value FF"
            $ps = str_repeat("\xFF", $psLen);
        }
        $em = chr(0) . chr($type) . $ps . chr(0) . $m;

        // RSA encryption
        $m = $this->_os2ip($em);
        $c = $this->_rsaep($m);
        $c = $this->_i2osp($c, $this->k);

        // Output the ciphertext C

        return $c;
    }

    /**
     * RSAEP.
     *
     * See {@link http://tools.ietf.org/html/rfc3447#section-5.1.1 RFC3447#section-5.1.1}.
     *
     * @param Math_BigInteger $m
     *
     * @return Math_BigInteger
     */
    public function _rsaep($m)
    {
        if ($m->compare($this->zero) < 0 || $m->compare($this->modulus) > 0) {
            trigger_error('Message representative out of range');

            return false;
        }

        return $this->_exponentiate($m);
    }

    /**
     * RSAES-OAEP-ENCRYPT.
     *
     * See {@link http://tools.ietf.org/html/rfc3447#section-7.1.1 RFC3447#section-7.1.1} and
     * {http://en.wikipedia.org/wiki/Optimal_Asymmetric_Encryption_Padding OAES}.
     *
     * @param string $m
     * @param string $l
     *
     * @return string
     */
    public function _rsaes_oaep_encrypt($m, $l = '')
    {
        $mLen = strlen($m);

        // Length checking

        // if $l is larger than two million terrabytes and you're using sha1, PKCS#1 suggests a "Label too long" error
        // be output.

        if ($mLen > $this->k - 2 * $this->hLen - 2) {
            trigger_error('Message too long');

            return false;
        }

        // EME-OAEP encoding

        $lHash      = $this->hash->hash($l);
        $ps         = str_repeat(chr(0), $this->k - $mLen - 2 * $this->hLen - 2);
        $db         = $lHash . $ps . chr(1) . $m;
        $seed       = crypt_random_string($this->hLen);
        $dbMask     = $this->_mgf1($seed, $this->k - $this->hLen - 1);
        $maskedDB   = $db ^ $dbMask;
        $seedMask   = $this->_mgf1($maskedDB, $this->hLen);
        $maskedSeed = $seed ^ $seedMask;
        $em         = chr(0) . $maskedSeed . $maskedDB;

        // RSA encryption

        $m = $this->_os2ip($em);
        $c = $this->_rsaep($m);
        $c = $this->_i2osp($c, $this->k);

        // Output the ciphertext C

        return $c;
    }

    /**
     * Decryption.
     *
     * @see encrypt()
     *
     * @return string
     */
    public function decrypt($ciphertext)
    {
        if ($this->k <= 0) {
            return false;
        }

        $ciphertext                         = str_split((string) $ciphertext, $this->k);
        $ciphertext[count($ciphertext) - 1] = str_pad($ciphertext[count($ciphertext) - 1], $this->k, chr(0), \STR_PAD_LEFT);

        $plaintext = '';

        $decrypt = match ($this->encryptionMode) {
            CRYPT_RSA_ENCRYPTION_PKCS1 => '_rsaes_pkcs1_v1_5_decrypt',
            default                    => '_rsaes_oaep_decrypt',
        };

        foreach ($ciphertext as $c) {
            $temp = $this->$decrypt($c);
            if (false === $temp) {
                return false;
            }
            $plaintext .= $temp;
        }

        return $plaintext;
    }

    /**
     * Create a signature.
     *
     * @see verify()
     *
     * @param string $message
     *
     * @return string
     */
    public function sign($message)
    {
        if (empty($this->modulus) || empty($this->exponent)) {
            return false;
        }

        return match ($this->signatureMode) {
            CRYPT_RSA_SIGNATURE_PKCS1 => $this->_rsassa_pkcs1_v1_5_sign($message),
            default                   => $this->_rsassa_pss_sign($message),
        };
    }

    /**
     * RSASSA-PKCS1-V1_5-SIGN.
     *
     * See {@link http://tools.ietf.org/html/rfc3447#section-8.2.1 RFC3447#section-8.2.1}.
     *
     * @param string $m
     *
     * @return string
     */
    public function _rsassa_pkcs1_v1_5_sign($m)
    {
        // EMSA-PKCS1-v1_5 encoding

        $em = $this->_emsa_pkcs1_v1_5_encode($m, $this->k);
        if (false === $em) {
            trigger_error('RSA modulus too short');

            return false;
        }

        // RSA signature

        $m = $this->_os2ip($em);
        $s = $this->_rsasp1($m);
        $s = $this->_i2osp($s, $this->k);

        // Output the signature S

        return $s;
    }

    /**
     * EMSA-PKCS1-V1_5-ENCODE.
     *
     * See {@link http://tools.ietf.org/html/rfc3447#section-9.2 RFC3447#section-9.2}.
     *
     * @param string $m
     * @param int    $emLen
     *
     * @return string
     */
    public function _emsa_pkcs1_v1_5_encode($m, $emLen)
    {
        $t = null;
        $h = $this->hash->hash($m);
        if (false === $h) {
            return false;
        }

        // see http://tools.ietf.org/html/rfc3447#page-43
        switch ($this->hashName) {
            case 'md2':
                $t = pack('H*', '3020300c06082a864886f70d020205000410');
                break;
            case 'md5':
                $t = pack('H*', '3020300c06082a864886f70d020505000410');
                break;
            case 'sha1':
                $t = pack('H*', '3021300906052b0e03021a05000414');
                break;
            case 'sha256':
                $t = pack('H*', '3031300d060960864801650304020105000420');
                break;
            case 'sha384':
                $t = pack('H*', '3041300d060960864801650304020205000430');
                break;
            case 'sha512':
                $t = pack('H*', '3051300d060960864801650304020305000440');
        }
        $t .= $h;
        $tLen = strlen($t);

        if ($emLen < $tLen + 11) {
            trigger_error('Intended encoded message length too short');

            return false;
        }

        $ps = str_repeat(chr(0xFF), $emLen - $tLen - 3);

        $em = "\0\1$ps\0$t";

        return $em;
    }

    /**
     * RSASP1.
     *
     * See {@link http://tools.ietf.org/html/rfc3447#section-5.2.1 RFC3447#section-5.2.1}.
     *
     * @param Math_BigInteger $m
     *
     * @return Math_BigInteger
     */
    public function _rsasp1($m)
    {
        if ($m->compare($this->zero) < 0 || $m->compare($this->modulus) > 0) {
            trigger_error('Message representative out of range');

            return false;
        }

        return $this->_exponentiate($m);
    }

    /**
     * RSASSA-PSS-SIGN.
     *
     * See {@link http://tools.ietf.org/html/rfc3447#section-8.1.1 RFC3447#section-8.1.1}.
     *
     * @param string $m
     *
     * @return string
     */
    public function _rsassa_pss_sign($m)
    {
        // EMSA-PSS encoding

        $em = $this->_emsa_pss_encode($m, 8 * $this->k - 1);

        // RSA signature

        $m = $this->_os2ip($em);
        $s = $this->_rsasp1($m);
        $s = $this->_i2osp($s, $this->k);

        // Output the signature S

        return $s;
    }

    /**
     * EMSA-PSS-ENCODE.
     *
     * See {@link http://tools.ietf.org/html/rfc3447#section-9.1.1 RFC3447#section-9.1.1}.
     *
     * @param string $m
     * @param int    $emBits
     */
    public function _emsa_pss_encode($m, $emBits)
    {
        // if $m is larger than two million terrabytes and you're using sha1, PKCS#1 suggests a "Label too long" error
        // be output.

        $emLen = ($emBits + 1) >> 3; // ie. ceil($emBits / 8)
        $sLen  = false == $this->sLen ? $this->hLen : $this->sLen;

        $mHash = $this->hash->hash($m);
        if ($emLen < $this->hLen + $sLen + 2) {
            trigger_error('Encoding error');

            return false;
        }

        $salt        = crypt_random_string($sLen);
        $m2          = "\0\0\0\0\0\0\0\0" . $mHash . $salt;
        $h           = $this->hash->hash($m2);
        $ps          = str_repeat(chr(0), $emLen - $sLen - $this->hLen - 2);
        $db          = $ps . chr(1) . $salt;
        $dbMask      = $this->_mgf1($h, $emLen - $this->hLen - 1);
        $maskedDB    = $db ^ $dbMask;
        $maskedDB[0] = ~chr(0xFF << ($emBits & 7)) & $maskedDB[0];
        $em          = $maskedDB . $h . chr(0xBC);

        return $em;
    }

    /**
     * Verifies a signature.
     *
     * @see sign()
     *
     * @param string $message
     * @param string $signature
     *
     * @return bool
     */
    public function verify($message, $signature)
    {
        if (empty($this->modulus) || empty($this->exponent)) {
            return false;
        }

        return match ($this->signatureMode) {
            CRYPT_RSA_SIGNATURE_PKCS1 => $this->_rsassa_pkcs1_v1_5_verify($message, $signature),
            default                   => $this->_rsassa_pss_verify($message, $signature),
        };
    }

    /**
     * RSASSA-PKCS1-V1_5-VERIFY.
     *
     * See {@link http://tools.ietf.org/html/rfc3447#section-8.2.2 RFC3447#section-8.2.2}.
     *
     * @param string $m
     *
     * @return string
     */
    public function _rsassa_pkcs1_v1_5_verify($m, $s)
    {
        // Length checking

        if (strlen((string) $s) != $this->k) {
            trigger_error('Invalid signature');

            return false;
        }

        // RSA verification

        $s  = $this->_os2ip($s);
        $m2 = $this->_rsavp1($s);
        if (false === $m2) {
            trigger_error('Invalid signature');

            return false;
        }
        $em = $this->_i2osp($m2, $this->k);
        if (false === $em) {
            trigger_error('Invalid signature');

            return false;
        }

        // EMSA-PKCS1-v1_5 encoding

        $em2 = $this->_emsa_pkcs1_v1_5_encode($m, $this->k);
        if (false === $em2) {
            trigger_error('RSA modulus too short');

            return false;
        }

        // Compare
        return $this->_equals($em, $em2);
    }

    /**
     * RSAVP1.
     *
     * See {@link http://tools.ietf.org/html/rfc3447#section-5.2.2 RFC3447#section-5.2.2}.
     *
     * @param Math_BigInteger $s
     *
     * @return Math_BigInteger
     */
    public function _rsavp1($s)
    {
        if ($s->compare($this->zero) < 0 || $s->compare($this->modulus) > 0) {
            trigger_error('Signature representative out of range');

            return false;
        }

        return $this->_exponentiate($s);
    }

    /**
     * Performs blinded RSA equality testing.
     *
     * Protects against a particular type of timing attack described.
     *
     * See {@link http://codahale.com/a-lesson-in-timing-attacks/ A Lesson In Timing Attacks (or, Don't use
     * MessageDigest.isEquals)}
     *
     * Thanks for the heads up singpolyma!
     *
     * @param string $x
     * @param string $y
     *
     * @return bool
     */
    public function _equals($x, $y)
    {
        if (strlen($x) != strlen($y)) {
            return false;
        }

        $result = 0;
        for ($i = 0; $i < strlen($x); ++$i) {
            $result |= ord($x[$i]) ^ ord($y[$i]);
        }

        return 0 == $result;
    }

    /**
     * RSASSA-PSS-VERIFY.
     *
     * See {@link http://tools.ietf.org/html/rfc3447#section-8.1.2 RFC3447#section-8.1.2}.
     *
     * @param string $m
     * @param string $s
     *
     * @return string
     */
    public function _rsassa_pss_verify($m, $s)
    {
        // Length checking

        if (strlen($s) != $this->k) {
            trigger_error('Invalid signature');

            return false;
        }

        // RSA verification

        $modBits = 8 * $this->k;

        $s2 = $this->_os2ip($s);
        $m2 = $this->_rsavp1($s2);
        if (false === $m2) {
            trigger_error('Invalid signature');

            return false;
        }
        $em = $this->_i2osp($m2, $modBits >> 3);
        if (false === $em) {
            trigger_error('Invalid signature');

            return false;
        }

        // EMSA-PSS verification

        return $this->_emsa_pss_verify($m, $em, $modBits - 1);
    }

    /**
     * EMSA-PSS-VERIFY.
     *
     * See {@link http://tools.ietf.org/html/rfc3447#section-9.1.2 RFC3447#section-9.1.2}.
     *
     * @param string $m
     * @param string $em
     * @param int    $emBits
     *
     * @return string
     */
    public function _emsa_pss_verify($m, $em, $emBits)
    {
        // if $m is larger than two million terrabytes and you're using sha1, PKCS#1 suggests a "Label too long" error
        // be output.

        $emLen = ($emBits + 1) >> 3; // ie. ceil($emBits / 8);
        $sLen  = false == $this->sLen ? $this->hLen : $this->sLen;

        $mHash = $this->hash->hash($m);
        if ($emLen < $this->hLen + $sLen + 2) {
            return false;
        }

        if ($em[strlen($em) - 1] != chr(0xBC)) {
            return false;
        }

        $maskedDB = substr($em, 0, -$this->hLen - 1);
        $h        = substr($em, -$this->hLen - 1, $this->hLen);
        $temp     = chr(0xFF << ($emBits & 7));
        if ((~$maskedDB[0] & $temp) != $temp) {
            return false;
        }
        $dbMask = $this->_mgf1($h, $emLen - $this->hLen - 1);
        $db     = $maskedDB ^ $dbMask;
        $db[0]  = ~chr(0xFF << ($emBits & 7)) & $db[0];
        $temp   = $emLen - $this->hLen - $sLen - 2;
        if (substr($db, 0, $temp) != str_repeat(chr(0), $temp) || 1 != ord($db[$temp])) {
            return false;
        }
        $salt = substr($db, $temp + 1); // should be $sLen long
        $m2   = "\0\0\0\0\0\0\0\0" . $mHash . $salt;
        $h2   = $this->hash->hash($m2);

        return $this->_equals($h, $h2);
    }
}
