<?php
/*
 * @package   bfNetwork
 * @copyright Copyright (C) 2011,2012,2013,2014,2015,2016,2017,2018,2019,2020,2021,2022,2023 Blue Flame Digital Solutions Ltd. All rights reserved.
 * @license   GNU General Public License version 3 or later
 *
 * @see       https://mySites.guru/
 * @see       https://www.phil-taylor.com/
 *
 * @author    Phil Taylor / Blue Flame Digital Solutions Limited.
 *
 * bfNetwork is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * bfNetwork is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this package.  If not, see http://www.gnu.org/licenses/
 *
 * If you have any questions regarding this code, please contact phil@phil-taylor.com
 */

/*
 * Include Crypt_Base.
 *
 * Base cipher class
 */
if (! class_exists('Crypt_Base')) {
    include_once 'Base.php';
}

/**#@+
 * @access private
 * @see Crypt_RC4::Crypt_RC4()
 */
/*
 * Toggles the internal implementation
 */
define('CRYPT_RC4_MODE_INTERNAL', CRYPT_MODE_INTERNAL);
/*
 * Toggles the mcrypt implementation
 */
define('CRYPT_RC4_MODE_MCRYPT', CRYPT_MODE_MCRYPT);
/**#@-*/

/**#@+
 * @access private
 * @see Crypt_RC4::_crypt()
 */
define('CRYPT_RC4_ENCRYPT', 0);
define('CRYPT_RC4_DECRYPT', 1);
/**#@-*/

/**
 * Pure-PHP implementation of RC4.
 *
 * @author  Jim Wigginton <terrafrost@php.net>
 */
class Crypt_RC4 extends Crypt_Base
{
    /**
     * Block Length of the cipher.
     *
     * RC4 is a stream cipher so we the block_size to 0
     *
     * @see Crypt_Base::block_size
     *
     * @var int
     */
    public $block_size = 0;

    /**
     * The default password key_size used by setPassword().
     *
     * @see Crypt_Base::password_key_size
     * @see Crypt_Base::setPassword()
     *
     * @var int
     */
    public $password_key_size = 128; // = 1024 bits

    /**
     * The namespace used by the cipher for its constants.
     *
     * @see Crypt_Base::const_namespace
     *
     * @var string
     */
    public $const_namespace = 'RC4';

    /**
     * The mcrypt specific name of the cipher.
     *
     * @see Crypt_Base::cipher_name_mcrypt
     *
     * @var string
     */
    public $cipher_name_mcrypt = 'arcfour';

    /**
     * Holds whether performance-optimized $inline_crypt() can/should be used.
     *
     * @see Crypt_Base::inline_crypt
     */
    public $use_inline_crypt = false; // currently not available

    /**
     * The Key.
     *
     * @see Crypt_RC4::setKey()
     *
     * @var string
     */
    public $key = "\0";

    /**
     * The Key Stream for decryption and encryption.
     *
     * @see Crypt_RC4::setKey()
     *
     * @var array
     */
    public $stream;

    /**
     * Default Constructor.
     *
     * Determines whether or not the mcrypt extension should be used.
     *
     * @see Crypt_Base::Crypt_Base()
     *
     * @return Crypt_RC4
     */
    public function __construct()
    {
        parent::__construct(CRYPT_MODE_STREAM);
    }

    /**
     * Dummy function.
     *
     * Some protocols, such as WEP, prepend an "initialization vector" to the key, effectively creating a new key [1].
     * If you need to use an initialization vector in this manner, feel free to prepend it to the key, yourself, before
     * calling setKey().
     *
     * [1] WEP's initialization vectors (IV's) are used in a somewhat insecure way.  Since, in that protocol, the IV's
     * are relatively easy to predict, an attack described by
     *
     * {@link http://www.drizzle.com/~aboba/IEEE/rc4_ksaproc.pdf Scott Fluhrer, Itsik Mantin, and Adi Shamir}
     * can be used to quickly guess at the rest of the key.  The following links elaborate:
     *
     * {@link http://www.rsa.com/rsalabs/node.asp?id=2009 http://www.rsa.com/rsalabs/node.asp?id=2009}
     * {@link http://en.wikipedia.org/wiki/Related_key_attack http://en.wikipedia.org/wiki/Related_key_attack}
     *
     * @param string $iv
     *
     * @see Crypt_RC4::setKey()
     */
    public function setIV($iv)
    {
    }

    /**
     * Sets the key.
     *
     * Keys can be between 1 and 256 bytes long.  If they are longer then 256 bytes, the first 256 bytes will be used.
     * If no key is explicitly set, it'll be assumed to be a single null byte.
     *
     * @see Crypt_Base::setKey()
     *
     * @param string $key
     */
    public function setKey($key)
    {
        parent::setKey(substr($key, 0, 256));
    }

    /**
     * Encrypts a message.
     *
     * @see Crypt_Base::decrypt()
     * @see Crypt_RC4::_crypt()
     *
     * @param string $plaintext
     *
     * @return string
     */
    public function encrypt($plaintext)
    {
        if (CRYPT_MODE_MCRYPT == $this->engine) {
            return parent::encrypt($plaintext);
        }

        return $this->_crypt($plaintext, CRYPT_RC4_ENCRYPT);
    }

    /**
     * Encrypts or decrypts a message.
     *
     * @see Crypt_RC4::encrypt()
     * @see Crypt_RC4::decrypt()
     *
     * @param string $text
     * @param int    $mode
     *
     * @return string
     */
    public function _crypt($text, $mode)
    {
        if ($this->changed) {
            $this->_setup();
            $this->changed = false;
        }

        $stream = &$this->stream[$mode];
        if ($this->continuousBuffer) {
            $i         = &$stream[0];
            $j         = &$stream[1];
            $keyStream = &$stream[2];
        } else {
            $i         = $stream[0];
            $j         = $stream[1];
            $keyStream = $stream[2];
        }

        $len = strlen($text);
        for ($k = 0; $k < $len; ++$k) {
            $i   = ($i + 1) & 255;
            $ksi = $keyStream[$i];
            $j   = ($j + $ksi) & 255;
            $ksj = $keyStream[$j];

            $keyStream[$i] = $ksj;
            $keyStream[$j] = $ksi;
            $text[$k]      = $text[$k] ^ chr($keyStream[($ksj + $ksi) & 255]);
        }

        return $text;
    }

    /**
     * Decrypts a message.
     *
     * $this->decrypt($this->encrypt($plaintext)) == $this->encrypt($this->encrypt($plaintext)). At least if the
     * continuous buffer is disabled.
     *
     * @see Crypt_Base::encrypt()
     * @see Crypt_RC4::_crypt()
     *
     * @param string $ciphertext
     *
     * @return string
     */
    public function decrypt($ciphertext)
    {
        if (CRYPT_MODE_MCRYPT == $this->engine) {
            return parent::decrypt($ciphertext);
        }

        return $this->_crypt($ciphertext, CRYPT_RC4_DECRYPT);
    }

    /**
     * Setup the key (expansion).
     *
     * @see Crypt_Base::_setupKey()
     */
    public function _setupKey()
    {
        $key       = $this->key;
        $keyLength = strlen($key);
        $keyStream = range(0, 255);
        $j         = 0;
        for ($i = 0; $i < 256; ++$i) {
            $j             = ($j + $keyStream[$i] + ord($key[$i % $keyLength])) & 255;
            $temp          = $keyStream[$i];
            $keyStream[$i] = $keyStream[$j];
            $keyStream[$j] = $temp;
        }

        $this->stream                    = [];
        $this->stream[CRYPT_RC4_DECRYPT] = $this->stream[CRYPT_RC4_ENCRYPT] = [
            0, // index $i
            0, // index $j
            $keyStream,
        ];
    }
}
