/**
 * @package   Tabs & Sliders
 * @contact   www.joomlashack.com, help@joomlashack.com
 * @copyright 2006-2015 JoomlaWorks Ltd. All rights reserved.
 * @copyright 2016-2024 Joomlashack. All rights reserved
 * @license   https://www.gnu.org/licenses/gpl.html GNU/GPL
 *
 * This file is part of Tabs & Sliders.
 *
 * Tabs & Sliders is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tabs & Sliders is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tabs & Sliders.  If not, see <https://www.gnu.org/licenses/>.
 */

/* Copyright (c) Patrick Fitzgerald - Version 1.9 */
;jQuery(document).ready(function($) {
    let tabberOptions = {
        'manualStartup': true,
        'cookie'       : "jwTabsCookie",
        'onLoad'       : function(argsObj) {
            let t = argsObj.tabber,
                i;
            if (t.id) {
                t.cookie = t.id + t.cookie;
            }
            i = parseInt(getCookie(t.cookie));
            if (isNaN(i)) {
                return;
            }
            t.tabShow(i);
        },
        'onClick'      : function(argsObj) {
            let c = argsObj.tabber.cookie,
                i = argsObj.index;
            setCookie(c, i);
        }
    };

    let setCookie = function(name, value, expires, path, domain, secure) {
        document.cookie = name + '=' + encodeURIComponent(value)
            + ((expires) ? '; expires=' + expires.toUTCString() : '')
            + ((path) ? '; path=' + path : '')
            + ((domain) ? '; domain=' + domain : '')
            + ((secure) ? '; secure' : '');
    }

    let getCookie = function(name) {
        let dc     = document.cookie,
            prefix = name + '=',
            begin  = dc.indexOf('; ' + prefix);

        if (begin === -1) {
            begin = dc.indexOf(prefix);
            if (begin !== 0) return null;
        } else {
            begin += 2;
        }

        let end = document.cookie.indexOf(';', begin);
        if (end === -1) {
            end = dc.length;
        }

        return decodeURIComponent(dc.substring(begin + prefix.length, end));
    }

    let deleteCookie = function(name, path, domain) {
        if (getCookie(name)) {
            document.cookie = name + '='
                + ((path) ? '; path=' + path : '')
                + ((domain) ? '; domain=' + domain : '')
                + '; expires=Thu, 01-Jan-70 00:00:01 GMT';
        }
    }

    let tabberObj = function(argsObj) {
        let arg;
        this.div                    = null;
        this.onLoad                 = null;
        this.onTabDisplay           = null;
        this.classMain              = 'jwts_tabber';
        this.classMainLive          = 'jwts_tabberlive';
        this.classTab               = 'jwts_tabbertab';
        this.classTabDefault        = 'jwts_tabbertabdefault';
        this.classNav               = 'jwts_tabbernav';
        this.classTabHide           = 'jwts_tabbertabhide';
        this.classNavActive         = 'jwts_tabberactive';
        this.titleElements          = ['h2', 'h3', 'h4', 'h5', 'h6'];
        this.titleElementsStripHTML = true;
        this.removeTitle            = true;
        this.addLinkId              = false;
        this.linkIdFormat           = '<tabberid>nav<tabnumberone>';
        for (arg in argsObj) {
            this[arg] = argsObj[arg];
        }
        this.REclassMain       = new RegExp('\\b' + this.classMain + '\\b', 'gi');
        this.REclassMainLive   = new RegExp('\\b' + this.classMainLive + '\\b', 'gi');
        this.REclassTab        = new RegExp('\\b' + this.classTab + '\\b', 'gi');
        this.REclassTabDefault = new RegExp('\\b' + this.classTabDefault + '\\b', 'gi');
        this.REclassTabHide    = new RegExp('\\b' + this.classTabHide + '\\b', 'gi');
        this.tabs              = [];
        if (this.div) {
            this.init(this.div);
            this.div = null;
        }
    }

    tabberObj.prototype.init = function(e) {
        let childNodes, i, i2, t, defaultTab = 0,
            DOM_ul, DOM_li, DOM_a, aId, headingElement;
        if (!document.getElementsByTagName) {
            return false;
        }
        if (e.id) {
            this.id = e.id;
        }
        this.tabs.length = 0;
        childNodes       = e.childNodes;
        for (i = 0; i < childNodes.length; i++) {
            if (childNodes[i].className && childNodes[i].className.match(this.REclassTab)) {
                t = {
                    div: childNodes[i]
                };

                this.tabs[this.tabs.length] = t;
                if (childNodes[i].className.match(this.REclassTabDefault)) {
                    defaultTab = this.tabs.length - 1;
                }
            }
        }
        DOM_ul           = document.createElement('ul');
        DOM_ul.className = this.classNav;
        for (i = 0; i < this.tabs.length; i++) {
            t             = this.tabs[i];
            t.headingText = t.div.title;
            if (this.removeTitle) {
                t.div.title = '';
            }
            if (!t.headingText) {
                for (i2 = 0; i2 < this.titleElements.length; i2++) {
                    headingElement = t.div.getElementsByTagName(this.titleElements[i2])[0];
                    if (headingElement) {
                        t.headingText = headingElement.innerHTML;
                        if (this.titleElementsStripHTML) {
                            t.headingText.replace(/<br>/gi, ' ');
                            t.headingText = t.headingText.replace(/<[^>]+>/g, '');
                        }
                        break;
                    }
                }
            }
            if (!t.headingText) {
                t.headingText = i + 1;
            }
            DOM_li = document.createElement('li');
            t.li   = DOM_li;
            DOM_a  = document.createElement('a');
            DOM_a.appendChild(document.createTextNode(t.headingText));
            DOM_a.href        = 'javascript:void(null);';
            DOM_a.title       = t.headingText;
            DOM_a.onclick     = this.navClick;
            DOM_a.tabber      = this;
            DOM_a.tabberIndex = i;
            if (this.addLinkId && this.linkIdFormat) {
                aId      = this.linkIdFormat;
                aId      = aId.replace(/<tabberid>/gi, this.id);
                aId      = aId.replace(/<tabnumberzero>/gi, i);
                aId      = aId.replace(/<tabnumberone>/gi, (i + 1).toString());
                aId      = aId.replace(/<tabtitle>/gi, t.headingText.replace(/[^a-zA-Z0-9\-]/gi, ''));
                DOM_a.id = aId;
            }
            DOM_li.appendChild(DOM_a);
            DOM_ul.appendChild(DOM_li);
        }
        e.insertBefore(DOM_ul, e.firstChild);
        e.className = e.className.replace(this.REclassMain, this.classMainLive);
        this.tabShow(defaultTab);
        if (typeof this.onLoad == 'function') {
            this.onLoad({
                tabber: this
            });
        }
        return this;
    };

    tabberObj.prototype.navClick = function(event) {
        let a = this;
        if (!a.tabber) {
            return false;
        }

        let self        = a.tabber,
            tabberIndex = a.tabberIndex,
            onClickArgs = null;

        a.blur();
        if (typeof self.onClick == 'function') {
            onClickArgs = {
                'tabber': self,
                'index' : tabberIndex,
                'event' : event
            };
            if (!event) {
                onClickArgs.event = window.event;
            }

            self.onClick(onClickArgs);
        }

        self.tabShow(tabberIndex);

        return false;
    };

    tabberObj.prototype.tabHideAll = function() {
        for (let i = 0; i < this.tabs.length; i++) {
            this.tabHide(i);
        }
    };

    tabberObj.prototype.tabHide = function(tabberIndex) {
        if (!this.tabs[tabberIndex]) {
            return false;
        }

        let div = this.tabs[tabberIndex].div;
        if (!div.className.match(this.REclassTabHide)) {
            div.className += ' ' + this.classTabHide;
        }

        this.navClearActive(tabberIndex);

        return this;
    };

    tabberObj.prototype.tabShow = function(tabberIndex) {
        if (!this.tabs[tabberIndex]) {
            return false;
        }

        this.tabHideAll();
        let div = this.tabs[tabberIndex].div;

        div.className = div.className.replace(this.REclassTabHide, '');
        this.navSetActive(tabberIndex);
        if (typeof this.onTabDisplay == 'function') {
            this.onTabDisplay({
                'tabber': this,
                'index' : tabberIndex
            });
        }

        return this;
    };

    tabberObj.prototype.navSetActive = function(tabberIndex) {
        this.tabs[tabberIndex].li.className = this.classNavActive;

        return this;
    };

    tabberObj.prototype.navClearActive = function(tabberIndex) {
        this.tabs[tabberIndex].li.className = '';

        return this;
    };

    let tabberAutomatic = function(tabberArgs) {
        let tempObj = new tabberObj(tabberArgs),
            divs    = document.getElementsByTagName('div');

        tabberArgs = tabberArgs || {};

        for (let i = 0; i < divs.length; i++) {
            if (divs[i].className && divs[i].className.match(tempObj.REclassMain)) {
                tabberArgs.div = divs[i];
                divs[i].tabber = new tabberObj(tabberArgs);
            }
        }

        return this;
    }

    let tabberAutomaticOnLoad = function(tabberArgs) {
        let oldOnLoad = window.onload;

        tabberArgs = tabberArgs || {};

        if (typeof window.onload != 'function') {
            window.onload = function() {
                tabberAutomatic(tabberArgs);
            };

        } else {
            window.onload = function() {
                oldOnLoad();
                tabberAutomatic(tabberArgs);
            };
        }
    }

    if (typeof tabberOptions == 'undefined') {
        tabberAutomaticOnLoad();

    } else {
        if (!tabberOptions['manualStartup']) {
            tabberAutomaticOnLoad(tabberOptions);
        }
    }

    // Tabs
    tabberAutomatic(tabberOptions);

    // Sliders
    $('.jwts_toggleContent').hide();
    $('a.jwts_toggleControl')
        .addClass('jwts_toggleOff')
        .on('click', function() {
            if ($('.jwts_toggleContent :animated').length) return false;

            let $this = $(this);
            if ($this.hasClass('jwts_toggleOn')) {
                $this.parent().next('.jwts_toggleContent').hide();
                $this
                    .removeClass('jwts_toggleOn')
                    .addClass('jwts_toggleOff');

            } else {
                $('.jwts_toggleContent').hide();
                $this.parent().next('.jwts_toggleContent').show();
                $('a.jwts_toggleControl')
                    .removeClass('jwts_toggleOn')
                    .addClass('jwts_toggleOff');
                $this.removeClass('jwts_toggleOff')
                    .addClass('jwts_toggleOn');
            }

            if (jsts_sliderAutoScroll) {
                $('html, body').animate({
                    scrollTop: $(this).offset().top
                }, 500);
            }

            return false;
        });
});
