<?php
/**
 * @package   onthos
 * @copyright Copyright (c) 2025 Nicholas K. Dionysopoulos / Akeeba Ltd
 * @license   GNU General Public License version 3, or later
 */

namespace Akeeba\Component\Onthos\Administrator\Library;

use Akeeba\Component\Onthos\Administrator\Library\Extension\ExtensionInterface;
use Joomla\CMS\Version;

defined('_JEXEC') || die;

abstract class CoreExtensions
{
	/**
	 * The internal cache of core Joomla extensions.
	 *
	 * @var   array
	 * @since 1.0.0
	 */
	static array $extensions;

	static string $maxVersion;

	public static function isCore(ExtensionInterface $extension): bool
	{
		$extensions = self::getCoreExtensions();

		return array_key_exists(self::getExtId($extension), $extensions);
	}

	public static function isSupported(ExtensionInterface $extension): bool
	{
		if (!self::isCore($extension))
		{
			return false;
		}

		$extensions = self::getCoreExtensions();
		$coreInfo   = $extensions[self::getExtId($extension)];
		$jVersion   = Version::MAJOR_VERSION . '.' . Version::MINOR_VERSION;

		return version_compare($jVersion, $coreInfo->min, 'ge')
		       && version_compare($jVersion, $coreInfo->max, 'le');
	}

	public static function isTooOld(ExtensionInterface $extension): bool
	{
		if (!self::isCore($extension))
		{
			return false;
		}

		$jVersion   = Version::MAJOR_VERSION . '.' . Version::MINOR_VERSION;
		$extensions = self::getCoreExtensions();
		$coreInfo   = $extensions[self::getExtId($extension)];

		return version_compare($jVersion, $coreInfo->max, 'gt');
	}

	public static function isTooNew(ExtensionInterface $extension): bool
	{
		if (!self::isCore($extension))
		{
			return false;
		}

		$extensions = self::getCoreExtensions();
		$coreInfo   = $extensions[self::getExtId($extension)];
		$jVersion   = Version::MAJOR_VERSION . '.' . Version::MINOR_VERSION;

		return version_compare($jVersion, $coreInfo->min, 'lt');
	}

	public static function minMaxVersions(ExtensionInterface $extension): array
	{
		$jVersion = Version::MAJOR_VERSION . '.' . Version::MINOR_VERSION;

		if (!self::isCore($extension))
		{
			return [$jVersion, $jVersion];
		}

		$extensions = self::getCoreExtensions();
		$coreInfo   = $extensions[self::getExtId($extension)];

		return [$coreInfo->min, $coreInfo->max];
	}

	public static function getMaxIncludedVersion(): string
	{
		return self::$maxVersion;
	}

	private static function getExtId(object $extension)
	{
		$type      = $extension->type;
		$element   = $extension->element;
		$folder    = $extension->folder;
		$client_id = $extension->client_id;

		return md5($type . '#' . $element . '#' . $folder . '#' . $client_id);
	}

	/**
	 * Returns the list of known core extensions.
	 *
	 * @return  array
	 * @since   1.0.0
	 */
	private static function getCoreExtensions(): array
	{
		if (isset(self::$extensions) && is_array(self::$extensions))
		{
			return self::$extensions;
		}

		self::$extensions = [];
		self::$maxVersion = Version::MAJOR_VERSION . '.' . Version::MINOR_VERSION;

		$json = @file_get_contents(__DIR__ . '/../joomla_extensions.json');

		if ($json === false)
		{
			return [];
		}

		try
		{
			$extensions = json_decode($json, false, JSON_THROW_ON_ERROR);
		}
		catch (\Exception $e)
		{
			$extensions = [];
		}

		$ids = array_map(self::getExtId(...), $extensions);

		self::$extensions = array_combine($ids, $extensions);
		self::$maxVersion = array_reduce(
			self::$extensions,
			fn($carry, $item) => version_compare($carry, $item->max, 'lt') ? $item->max : $carry,
			'0.0'
		);

		$jVersion = Version::MAJOR_VERSION . '.' . Version::MINOR_VERSION;

		/**
		 * If our core information stops at an earlier Joomla version we will change all max supported versions equal to
		 * that version to the current one.
		 *
		 * This is necessary to not mark every core extension as obsolete, recommending that the user deletes their
		 * entire freaking site.
		 */
		if (version_compare($jVersion, self::$maxVersion, 'gt'))
		{
			self::$extensions = array_map(
				function ($item) use ($jVersion) {
					if ($item->max === self::$maxVersion)
					{
						$item->max = $jVersion;
					}

					return $item;
				},
				self::$extensions,
			);
		}

		return self::$extensions;
	}
}